<?php
// core/SupportManager.php
class SupportManager {
    private $dataFile;
    
    public function __construct() {
        $this->dataFile = dirname(__DIR__) . '/data/support_tickets.json';
        $this->initData();
    }
    
    private function initData() {
        if (!file_exists($this->dataFile)) {
            $data = [
                'tickets' => [],
                'next_ticket_id' => 1,
                'statistics' => [
                    'total_tickets' => 0,
                    'answered_tickets' => 0,
                    'pending_tickets' => 0
                ]
            ];
            file_put_contents($this->dataFile, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
        }
    }
    
    private function loadData() {
        $json = file_get_contents($this->dataFile);
        return json_decode($json, true);
    }
    
    private function saveData($data) {
        file_put_contents($this->dataFile, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    }
    
    /**
     * ایجاد تیکت جدید
     */
    public function createTicket($user_id, $message, $category = 'عمومی') {
        $data = $this->loadData();
        
        $ticket_id = $data['next_ticket_id']++;
        $ticket_code = 'T' . date('Ymd') . str_pad($ticket_id, 4, '0', STR_PAD_LEFT);
        
        $ticket = [
            'id' => $ticket_id,
            'ticket_code' => $ticket_code,
            'user_id' => $user_id,
            'category' => $category,
            'message' => $message,
            'status' => 'open', // open, answered, closed
            'created_at' => date('Y-m-d H:i:s'),
            'answered_at' => null,
            'admin_id' => null,
            'response' => null
        ];
        
        $data['tickets'][$ticket_id] = $ticket;
        $data['statistics']['total_tickets']++;
        $data['statistics']['pending_tickets']++;
        
        $this->saveData($data);
        
        return [
            'success' => true,
            'ticket_id' => $ticket_id,
            'ticket_code' => $ticket_code
        ];
    }
    
    /**
     * دریافت تیکت‌های کاربر
     */
    public function getUserTickets($user_id, $limit = 5) {
        $data = $this->loadData();
        $user_tickets = [];
        
        foreach ($data['tickets'] as $ticket) {
            if ($ticket['user_id'] == $user_id) {
                $user_tickets[] = $ticket;
            }
        }
        
        // مرتب سازی بر اساس زمان (جدیدترین اول)
        usort($user_tickets, function($a, $b) {
            return strtotime($b['created_at']) - strtotime($a['created_at']);
        });
        
        return array_slice($user_tickets, 0, $limit);
    }
    
    /**
     * دریافت تیکت‌های باز
     */
    public function getOpenTickets() {
        $data = $this->loadData();
        $open_tickets = [];
        
        foreach ($data['tickets'] as $ticket) {
            if ($ticket['status'] == 'open') {
                $open_tickets[] = $ticket;
            }
        }
        
        return $open_tickets;
    }
    
    /**
     * پاسخ به تیکت
     */
    public function respondToTicket($ticket_id, $admin_id, $response) {
        $data = $this->loadData();
        
        if (isset($data['tickets'][$ticket_id])) {
            $data['tickets'][$ticket_id]['status'] = 'answered';
            $data['tickets'][$ticket_id]['answered_at'] = date('Y-m-d H:i:s');
            $data['tickets'][$ticket_id]['admin_id'] = $admin_id;
            $data['tickets'][$ticket_id]['response'] = $response;
            
            $data['statistics']['pending_tickets']--;
            $data['statistics']['answered_tickets']++;
            
            $this->saveData($data);
            return true;
        }
        
        return false;
    }
    
    /**
     * آمار تیکت‌ها
     */
    public function getStats() {
        $data = $this->loadData();
        return $data['statistics'];
    }
}
?>