<?php
// core/AnonLinkManager.php - سیستم ساده لینک ناشناس با JSON
class AnonLinkManager {
    private $dataFile;
    
    public function __construct() {
        $this->dataFile = dirname(__DIR__) . '/data/anon_links.json';
        $this->initData();
    }
    
    private function initData() {
        if (!file_exists($this->dataFile)) {
            $data = [
                'links' => [],
                'messages' => [],
                'statistics' => [
                    'total_links' => 0,
                    'total_messages' => 0
                ]
            ];
            file_put_contents($this->dataFile, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
        }
    }
    
    private function loadData() {
        $json = file_get_contents($this->dataFile);
        return json_decode($json, true);
    }
    
    private function saveData($data) {
        file_put_contents($this->dataFile, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    }
    
    // 🔗 ساخت لینک جدید
    public function createLink($user_id, $link_name = 'لینک ناشناس') {
        $data = $this->loadData();
        
        // تولید کد منحصر به فرد
        $link_code = 'AL' . time() . substr(md5($user_id . uniqid()), 0, 8);
        
        $link = [
            'id' => uniqid('', true),
            'user_id' => (string)$user_id,
            'link_code' => $link_code,
            'link_name' => $link_name,
            'created_at' => date('Y-m-d H:i:s'),
            'expires_at' => date('Y-m-d H:i:s', strtotime('+30 days')),
            'is_active' => true,
            'message_count' => 0,
            'views' => 0
        ];
        
        $data['links'][] = $link;
        $data['statistics']['total_links']++;
        $this->saveData($data);
        
        $telegram_link = "https://t.me/" . BOT_USERNAME . "?start=anon_{$link_code}";
        
        return [
            'success' => true,
            'link_code' => $link_code,
            'telegram_link' => $telegram_link,
            'expires_at' => $link['expires_at']
        ];
    }
    
    // 🔍 دریافت لینک با کد
    public function getLinkByCode($link_code) {
        $data = $this->loadData();
        
        foreach ($data['links'] as $link) {
            if ($link['link_code'] === $link_code && $link['is_active']) {
                // افزایش بازدید
                $this->increaseViews($link['id']);
                return $link;
            }
        }
        
        return null;
    }
    
    // 👁 افزایش بازدید
    private function increaseViews($link_id) {
        $data = $this->loadData();
        
        foreach ($data['links'] as &$link) {
            if ($link['id'] === $link_id) {
                $link['views']++;
                $this->saveData($data);
                break;
            }
        }
    }
    
    // 📨 ذخیره پیام ناشناس
    public function saveAnonymousMessage($link_code, $message, $message_type = 'text', $sender_ip = null) {
        $data = $this->loadData();
        $link_found = null;
        
        // یافتن لینک
        foreach ($data['links'] as &$link) {
            if ($link['link_code'] === $link_code && $link['is_active']) {
                $link_found = &$link;
                break;
            }
        }
        
        if (!$link_found) {
            return ['success' => false, 'error' => 'لینک نامعتبر'];
        }
        
        // ذخیره پیام
        $new_message = [
            'id' => uniqid('', true),
            'link_id' => $link_found['id'],
            'message' => $message,
            'sender_ip' => $sender_ip,
            'created_at' => date('Y-m-d H:i:s'),
            'is_read' => false
        ];
        
        $data['messages'][] = $new_message;
        
        // آپدیت آمار لینک
        $link_found['message_count']++;
        $data['statistics']['total_messages']++;
        
        $this->saveData($data);
        
        return [
            'success' => true,
            'message_id' => $new_message['id'],
            'link_owner_id' => (int)$link_found['user_id'],
            'link_name' => $link_found['link_name']
        ];
    }
    
    // 👤 دریافت لینک‌های کاربر
    public function getUserLinks($user_id, $active_only = true) {
        $data = $this->loadData();
        $user_links = [];
        
        foreach ($data['links'] as $link) {
            if ($link['user_id'] == (string)$user_id) {
                if (!$active_only || ($active_only && $link['is_active'])) {
                    $user_links[] = $link;
                }
            }
        }
        
        return $user_links;
    }
    
    // 📩 دریافت پیام‌های لینک
    public function getLinkMessages($link_id, $limit = 10) {
        $data = $this->loadData();
        $link_messages = [];
        $count = 0;
        
        foreach ($data['messages'] as $message) {
            if ($message['link_id'] === $link_id) {
                $link_messages[] = $message;
                $count++;
                if ($count >= $limit) break;
            }
        }
        
        // مرتب سازی جدیدترین اول
        usort($link_messages, function($a, $b) {
            return strtotime($b['created_at']) - strtotime($a['created_at']);
        });
        
        return $link_messages;
    }
    
    // 📊 دریافت آمار لینک
    public function getLinkStats($link_id) {
        $data = $this->loadData();
        
        $total_messages = 0;
        $read_messages = 0;
        
        foreach ($data['messages'] as $message) {
            if ($message['link_id'] === $link_id) {
                $total_messages++;
                if ($message['is_read']) {
                    $read_messages++;
                }
            }
        }
        
        return [
            'total_messages' => $total_messages,
            'read_messages' => $read_messages
        ];
    }
    
    // 📈 دریافت آمار کاربر
    public function getUserStats($user_id) {
        $links = $this->getUserLinks($user_id, true);
        $total_messages = 0;
        $total_views = 0;
        
        foreach ($links as $link) {
            $total_messages += $link['message_count'];
            $total_views += $link['views'];
        }
        
        return [
            'total_links' => count($links),
            'total_messages' => $total_messages,
            'total_views' => $total_views
        ];
    }
    public function cleanupOldStates() {
    // هر 10 استفاده، stateهای قدیمی رو پاک کن
    $usage_file = DATA_PATH . 'link_usage_count.txt';
    
    if (!file_exists($usage_file)) {
        file_put_contents($usage_file, '0');
        return;
    }
    
    $count = (int)file_get_contents($usage_file);
    $count++;
    
    if ($count >= 10) {
        // پاکسازی stateهای قدیمی
        $users_file = DATA_PATH . 'users.json';
        if (file_exists($users_file)) {
            $users = json_decode(file_get_contents($users_file), true);
            foreach ($users as &$user) {
                // فقط stateهای لینک ناشناس رو پاک کن
                $user['anon_link_code'] = null;
                $user['temp_anon_message'] = null;
            }
            file_put_contents($users_file, json_encode($users, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
        }
        $count = 0;
    }
    
    file_put_contents($usage_file, (string)$count);
}
}
?>