<?php
class UserManager {
    private $data;
    private $file;
    
    public function __construct() {
        $this->file = USERS_FILE;
        $this->loadData();
    }
    
    private function loadData() {
        if (!file_exists($this->file)) {
            $this->data = [
                'users' => [],
                'waiting_list' => [],
                'filtered_waiting' => [
                    'دختر' => [],
                    'پسر' => []
                ],
                'active_chats' => [],
                'blocked_users' => [],
                'refund_queue' => [],
                'likes_history' => [], // ✅ جدید
                'dislikes_history' => [], // ✅ جدید
                'statistics' => [
                    'total_users' => 0,
                    'active_today' => 0,
                    'total_chats' => 0,
                    'total_refunds' => 0
                ]
            ];
            $this->saveData();
        } else {
            $json_data = file_get_contents($this->file);
            $this->data = json_decode($json_data, true);
            
            if (!$this->data || !is_array($this->data)) {
                $this->data = [
                    'users' => [],
                    'waiting_list' => [],
                    'filtered_waiting' => [
                        'دختر' => [],
                        'پسر' => []
                    ],
                    'active_chats' => [],
                    'blocked_users' => [],
                    'refund_queue' => [],
                    'likes_history' => [],
                    'dislikes_history' => [],
                    'statistics' => [
                        'total_users' => 0,
                        'active_today' => 0,
                        'total_chats' => 0,
                        'total_refunds' => 0
                    ]
                ];
            }
            
            // اطمینان از وجود ساختارها
            if (!isset($this->data['refund_queue'])) {
                $this->data['refund_queue'] = [];
            }
            
            if (!isset($this->data['likes_history'])) {
                $this->data['likes_history'] = [];
            }
            
            if (!isset($this->data['dislikes_history'])) {
                $this->data['dislikes_history'] = [];
            }
            
            if (!isset($this->data['statistics']['total_refunds'])) {
                $this->data['statistics']['total_refunds'] = 0;
            }
        }
    }
    
    private function saveData() {
        $temp_file = $this->file . '.tmp';
        if (file_put_contents($temp_file, json_encode($this->data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE))) {
            if (rename($temp_file, $this->file)) {
                return true;
            }
        }
        return false;
    }
    
    public function createUser($user_id, $user_data) {
        $is_new_user = !isset($this->data['users'][$user_id]);
        
        if ($is_new_user) {
            $this->data['statistics']['total_users']++;
        }
        
        $default_data = [
            'user_id' => $user_id,
            'first_name' => '',
            'last_name' => '',
            'username' => '',
            'sex' => '',
            'age' => 0,
            'city' => '',
            'step' => 'none',
            'status' => 'active',
            'vip' => false,
            'likes' => 0,
            'dislikes' => 0,
            'invited_count' => 0,
            'chat_count' => 0,
            'coins' => 0,
            'created_at' => date('Y-m-d H:i:s'),
            'last_active' => date('Y-m-d H:i:s'),
            'current_chat' => null,
            'partner_id' => null,
            'welcome_coins_given' => false,
            'preferred_gender' => null,
            'chat_mode' => null,
            'display_name' => ''
        ];
        
        if ($is_new_user) {
            $this->data['users'][$user_id] = array_merge($default_data, $user_data);
        } else {
            $this->data['users'][$user_id] = array_merge($this->data['users'][$user_id], $user_data);
        }
        
        $this->data['users'][$user_id]['last_active'] = date('Y-m-d H:i:s');
        
        return $this->saveData();
    }
    
    public function getUser($user_id) {
        return $this->data['users'][$user_id] ?? null;
    }
    
    public function updateUser($user_id, $updates) {
        if (isset($this->data['users'][$user_id])) {
            $this->data['users'][$user_id] = array_merge($this->data['users'][$user_id], $updates);
            $this->data['users'][$user_id]['last_active'] = date('Y-m-d H:i:s');
            return $this->saveData();
        }
        return false;
    }
    
    public function giveWelcomeCoins($user_id) {
        $user = $this->getUser($user_id);
        if (!$user) return false;
        
        if (!empty($user['sex']) && !empty($user['age']) && !empty($user['city']) && 
            !empty($user['display_name']) && !($user['welcome_coins_given'] ?? false)) {
            
            $this->updateUser($user_id, [
                'coins' => ($user['coins'] ?? 0) + 5,
                'welcome_coins_given' => true
            ]);
            return true;
        }
        return false;
    }
    
    public function addCoins($user_id, $coins, $reason = '') {
        $user = $this->getUser($user_id);
        if (!$user) return false;
        
        $current_coins = $user['coins'] ?? 0;
        $new_coins = $current_coins + $coins;
        
        return $this->updateUser($user_id, [
            'coins' => $new_coins
        ]);
    }
    
    public function deductCoin($user_id) {
        $user = $this->getUser($user_id);
        if (!$user) return false;
        
        $current_coins = $user['coins'] ?? 0;
        
        if ($current_coins >= 1) {
            return $this->updateUser($user_id, [
                'coins' => $current_coins - 1
            ]);
        }
        return false;
    }
    
    public function getCoins($user_id) {
        $user = $this->getUser($user_id);
        return $user['coins'] ?? 0;
    }
    
    public function generateInviteLink($user_id) {
        $user = $this->getUser($user_id);
        if (!$user) return false;
        
        $invite_code = substr(md5($user_id . 'secret_salt'), 0, 8);
        $this->updateUser($user_id, ['invite_code' => $invite_code]);
        
        return "https://t.me/jobaychatbot?start=invite_{$invite_code}";
    }
    
    public function handleInvite($invited_user_id, $invite_code) {
        foreach ($this->data['users'] as $user) {
            if (($user['invite_code'] ?? '') === $invite_code) {
                $inviter_id = $user['user_id'];
                
                $this->updateUser($inviter_id, [
                    'invited_count' => ($user['invited_count'] ?? 0) + 1
                ]);
                
                $this->addCoins($inviter_id, 10, "دعوت کاربر");
                $this->addCoins($invited_user_id, 10, "عضویت از طریق دعوت");
                
                $this->updateUser($invited_user_id, [
                    'invited_by' => $inviter_id
                ]);
                
                return [
                    'inviter_id' => $inviter_id,
                    'inviter_name' => $user['first_name'] . ' ' . $user['last_name'],
                    'invited_user_id' => $invited_user_id,
                    'success' => true
                ];
            }
        }
        return ['success' => false];
    }
    
    public function addToWaitingList($user_id) {
        $this->createUser($user_id, []);
        
        if (!in_array($user_id, $this->data['waiting_list'])) {
            $this->data['waiting_list'][] = $user_id;
            $this->updateUser($user_id, [
                'status' => 'waiting',
                'chat_mode' => 'random'
            ]);
            return $this->saveData();
        }
        return true;
    }
    
    public function removeFromWaitingList($user_id) {
        $key = array_search($user_id, $this->data['waiting_list']);
        if ($key !== false) {
            unset($this->data['waiting_list'][$key]);
            $this->data['waiting_list'] = array_values($this->data['waiting_list']);
            $this->updateUser($user_id, [
                'status' => 'active',
                'chat_mode' => null
            ]);
            return $this->saveData();
        }
        return true;
    }
    
    public function getWaitingList() {
        $active_users = [];
        $five_minutes_ago = time() - 300;
        
        foreach ($this->data['waiting_list'] as $user_id) {
            $user = $this->getUser($user_id);
            if ($user && strtotime($user['last_active']) >= $five_minutes_ago) {
                $active_users[] = $user_id;
            }
        }
        
        $this->data['waiting_list'] = $active_users;
        $this->saveData();
        
        return $active_users;
    }
    
    public function requestFilteredChat($user_id, $preferred_gender) {
        $user = $this->getUser($user_id);
        if (!$user) return ['success' => false, 'error' => 'کاربر یافت نشد'];
        
        if (!$user['vip']) {
            $current_coins = $user['coins'] ?? 0;
            if ($current_coins < 2) {
                return ['success' => false, 'error' => 'کمبود سکه', 'coins_needed' => 2];
            }
            $this->updateUser($user_id, ['coins' => $current_coins - 2]);
        }
        
        $this->updateUser($user_id, [
            'preferred_gender' => $preferred_gender,
            'chat_mode' => 'filtered',
            'status' => 'waiting_filtered',
            'last_chat_request' => time()
        ]);
        
        $user_gender = $user['sex'];
        
        if (!isset($this->data['filtered_waiting'][$user_gender])) {
            $this->data['filtered_waiting'][$user_gender] = [];
        }
        
        if (!in_array($user_id, $this->data['filtered_waiting'][$user_gender])) {
            $this->data['filtered_waiting'][$user_gender][] = $user_id;
        }
        
        $this->saveData();
        
        return ['success' => true, 'coins_deducted' => (!$user['vip'] ? 2 : 0)];
    }
    
    public function findFilteredPartner($user_id) {
        $user = $this->getUser($user_id);
        if (!$user || !isset($user['preferred_gender'])) {
            return false;
        }
        
        $preferred_gender = $user['preferred_gender'];
        $user_gender = $user['sex'];
        
        if (isset($this->data['filtered_waiting'][$preferred_gender])) {
            $waiting_users = $this->data['filtered_waiting'][$preferred_gender];
            
            $key = array_search($user_id, $waiting_users);
            if ($key !== false) {
                unset($waiting_users[$key]);
                $waiting_users = array_values($waiting_users);
                $this->data['filtered_waiting'][$preferred_gender] = $waiting_users;
            }
            
            if (!empty($waiting_users)) {
                $partner_id = $waiting_users[array_rand($waiting_users)];
                
                $partner_key = array_search($partner_id, $this->data['filtered_waiting'][$preferred_gender]);
                if ($partner_key !== false) {
                    unset($this->data['filtered_waiting'][$preferred_gender][$partner_key]);
                    $this->data['filtered_waiting'][$preferred_gender] = array_values($this->data['filtered_waiting'][$preferred_gender]);
                }
                
                $this->saveData();
                return $partner_id;
            }
        }
        
        if (!isset($this->data['filtered_waiting'][$user_gender])) {
            $this->data['filtered_waiting'][$user_gender] = [];
        }
        
        if (!in_array($user_id, $this->data['filtered_waiting'][$user_gender])) {
            $this->data['filtered_waiting'][$user_gender][] = $user_id;
            $this->saveData();
        }
        
        return false;
    }
    
    public function removeFromFilteredWaiting($user_id) {
        $user = $this->getUser($user_id);
        if (!$user) return false;
        
        $removed = false;
        
        foreach ($this->data['filtered_waiting'] as $gender => $users) {
            $key = array_search($user_id, $users);
            if ($key !== false) {
                unset($this->data['filtered_waiting'][$gender][$key]);
                $this->data['filtered_waiting'][$gender] = array_values($this->data['filtered_waiting'][$gender]);
                $removed = true;
            }
        }
        
        if ($removed) {
            $this->updateUser($user_id, [
                'status' => 'active',
                'preferred_gender' => null,
                'chat_mode' => null
            ]);
            $this->saveData();
            return true;
        }
        
        return false;
    }
    
    public function createChat($user1_id, $user2_id) {
        $this->createUser($user1_id, []);
        $this->createUser($user2_id, []);
        
        $user1 = $this->getUser($user1_id);
        $user2 = $this->getUser($user2_id);
        
        if (($user1['current_chat'] ?? null) || ($user2['current_chat'] ?? null)) {
            return false;
        }
        
        $chat_id = uniqid('chat_', true);
        
        $this->data['active_chats'][$chat_id] = [
            'user1' => $user1_id,
            'user2' => $user2_id,
            'created_at' => date('Y-m-d H:i:s'),
            'chat_start_time' => time(),
            'messages_count' => 0,
            'coins_deducted' => [
                $user1_id => ($user1['chat_mode'] == 'filtered' && !$user1['vip']) ? 2 : 0,
                $user2_id => ($user2['chat_mode'] == 'filtered' && !$user2['vip']) ? 2 : 0
            ]
        ];
        
        $this->updateUser($user1_id, [
            'status' => 'chatting',
            'current_chat' => $chat_id,
            'partner_id' => $user2_id,
            'preferred_gender' => null,
            'chat_mode' => null,
            'last_chat_start' => time()
        ]);
        
        $this->updateUser($user2_id, [
            'status' => 'chatting',
            'current_chat' => $chat_id,
            'partner_id' => $user1_id,
            'preferred_gender' => null,
            'chat_mode' => null,
            'last_chat_start' => time()
        ]);
        
        $this->data['statistics']['total_chats']++;
        
        $this->removeFromWaitingList($user1_id);
        $this->removeFromWaitingList($user2_id);
        $this->removeFromFilteredWaiting($user1_id);
        $this->removeFromFilteredWaiting($user2_id);
        
        return $this->saveData() ? $chat_id : false;
    }
    
    public function endChat($user_id) {
        $user = $this->getUser($user_id);
        
        if (!$user || !$user['current_chat']) {
            return false;
        }
        
        $chat_id = $user['current_chat'];
        
        if (!isset($this->data['active_chats'][$chat_id])) {
            $this->updateUser($user_id, [
                'status' => 'active',
                'current_chat' => null,
                'partner_id' => null
            ]);
            return true;
        }
        
        $chat = $this->data['active_chats'][$chat_id];
        $partner_id = ($chat['user1'] == $user_id) ? $chat['user2'] : $chat['user1'];
        
        $chat_duration = time() - $chat['chat_start_time'];
        
        if ($chat_duration < 10 && ($chat['coins_deducted'][$user_id] > 0 || $chat['coins_deducted'][$partner_id] > 0)) {
            $this->processRefund($chat, $chat_id);
        }
        
        $this->updateUser($user_id, [
            'status' => 'active',
            'current_chat' => null,
            'partner_id' => null,
            'chat_count' => ($user['chat_count'] ?? 0) + 1
        ]);
        
        if ($partner_id) {
            $partner = $this->getUser($partner_id);
            $this->updateUser($partner_id, [
                'status' => 'active',
                'current_chat' => null,
                'partner_id' => null,
                'chat_count' => ($partner['chat_count'] ?? 0) + 1
            ]);
        }
        
        unset($this->data['active_chats'][$chat_id]);
        
        return $this->saveData();
    }
    
    private function processRefund($chat, $chat_id) {
        $user1_id = $chat['user1'];
        $user2_id = $chat['user2'];
        
        $user1 = $this->getUser($user1_id);
        $user2 = $this->getUser($user2_id);
        
        if ($chat['coins_deducted'][$user1_id] > 0 && !$user1['vip']) {
            $new_coins = ($user1['coins'] ?? 0) + $chat['coins_deducted'][$user1_id];
            $this->updateUser($user1_id, [
                'coins' => $new_coins,
                'last_refund' => date('Y-m-d H:i:s'),
                'refund_amount' => $chat['coins_deducted'][$user1_id]
            ]);
            
            $this->data['refund_queue'][] = [
                'user_id' => $user1_id,
                'amount' => $chat['coins_deducted'][$user1_id],
                'chat_id' => $chat_id,
                'reason' => 'چت کوتاه مدت (کمتر از ۱۰ ثانیه)',
                'timestamp' => time()
            ];
        }
        
        if ($chat['coins_deducted'][$user2_id] > 0 && !$user2['vip']) {
            $new_coins = ($user2['coins'] ?? 0) + $chat['coins_deducted'][$user2_id];
            $this->updateUser($user2_id, [
                'coins' => $new_coins,
                'last_refund' => date('Y-m-d H:i:s'),
                'refund_amount' => $chat['coins_deducted'][$user2_id]
            ]);
            
            $this->data['refund_queue'][] = [
                'user_id' => $user2_id,
                'amount' => $chat['coins_deducted'][$user2_id],
                'chat_id' => $chat_id,
                'reason' => 'چت کوتاه مدت (کمتر از ۱۰ ثانیه)',
                'timestamp' => time()
            ];
        }
        
        $this->data['statistics']['total_refunds']++;
        $this->saveData();
    }
    
    public function getChatPartner($user_id) {
        $user = $this->getUser($user_id);
        
        if (!$user || !isset($user['current_chat'])) {
            return null;
        }
        
        $chat_id = $user['current_chat'];
        
        if (!isset($this->data['active_chats'][$chat_id])) {
            $this->updateUser($user_id, [
                'status' => 'active',
                'current_chat' => null,
                'partner_id' => null
            ]);
            return null;
        }
        
        $chat = $this->data['active_chats'][$chat_id];
        
        if ($chat['user1'] == $user_id) {
            return $chat['user2'];
        } elseif ($chat['user2'] == $user_id) {
            return $chat['user1'];
        }
        
        return null;
    }
    
    public function getActiveChat($user_id) {
        $user = $this->getUser($user_id);
        
        if (!$user || !$user['current_chat']) {
            return null;
        }
        
        $chat_id = $user['current_chat'];
        return $this->data['active_chats'][$chat_id] ?? null;
    }
    
    public function blockUser($user_id) {
        if (!in_array($user_id, $this->data['blocked_users'])) {
            $this->data['blocked_users'][] = $user_id;
            
            $user = $this->getUser($user_id);
            if ($user && $user['current_chat']) {
                $this->endChat($user_id);
            }
            
            $this->removeFromWaitingList($user_id);
            $this->removeFromFilteredWaiting($user_id);
            
            $this->updateUser($user_id, ['status' => 'blocked']);
            
            return $this->saveData();
        }
        return true;
    }
    
    public function unblockUser($user_id) {
        $key = array_search($user_id, $this->data['blocked_users']);
        if ($key !== false) {
            unset($this->data['blocked_users'][$key]);
            $this->data['blocked_users'] = array_values($this->data['blocked_users']);
            $this->updateUser($user_id, ['status' => 'active']);
            return $this->saveData();
        }
        return true;
    }
    
    public function isBlocked($user_id) {
        return in_array($user_id, $this->data['blocked_users']);
    }
    
    public function cleanupInactiveUsers() {
        $inactive_time = time() - 1800;
        $removed_count = 0;
        
        foreach ($this->data['waiting_list'] as $key => $user_id) {
            $user = $this->getUser($user_id);
            if (!$user || strtotime($user['last_active']) < $inactive_time) {
                unset($this->data['waiting_list'][$key]);
                $removed_count++;
            }
        }
        
        $this->data['waiting_list'] = array_values($this->data['waiting_list']);
        
        foreach ($this->data['filtered_waiting'] as $gender => $users) {
            foreach ($users as $key => $user_id) {
                $user = $this->getUser($user_id);
                if (!$user || strtotime($user['last_active']) < $inactive_time) {
                    unset($this->data['filtered_waiting'][$gender][$key]);
                    $removed_count++;
                }
            }
            $this->data['filtered_waiting'][$gender] = array_values($this->data['filtered_waiting'][$gender]);
        }
        
        if ($removed_count > 0) {
            $this->saveData();
        }
        
        return $removed_count;
    }
    
    public function getStatistics() {
        $this->data['statistics']['active_today'] = 0;
        $today = date('Y-m-d');
        
        foreach ($this->data['users'] as $user) {
            if (strpos($user['last_active'] ?? '', $today) === 0) {
                $this->data['statistics']['active_today']++;
            }
        }
        
        $this->saveData();
        
        return $this->data['statistics'];
    }
    
    public function getUserCount() {
        return count($this->data['users']);
    }
    
    public function getActiveChatsCount() {
        return count($this->data['active_chats']);
    }
    
    public function getFilteredWaitingStats() {
        return [
            'دختر' => count($this->data['filtered_waiting']['دختر'] ?? []),
            'پسر' => count($this->data['filtered_waiting']['پسر'] ?? [])
        ];
    }
    
    public function getPendingRefunds() {
        return $this->data['refund_queue'] ?? [];
    }
    
    public function removeRefundFromQueue($refund_index) {
        if (isset($this->data['refund_queue'][$refund_index])) {
            unset($this->data['refund_queue'][$refund_index]);
            $this->data['refund_queue'] = array_values($this->data['refund_queue']);
            return $this->saveData();
        }
        return false;
    }
    
    public function processAutoRefunds() {
        $now = time();
        $processed = 0;
        
        foreach ($this->data['refund_queue'] as $index => $refund) {
            if ($now - $refund['timestamp'] > 86400) {
                unset($this->data['refund_queue'][$index]);
                $processed++;
            }
        }
        
        if ($processed > 0) {
            $this->data['refund_queue'] = array_values($this->data['refund_queue']);
            $this->saveData();
        }
        
        return $processed;
    }
    
    // 🔥 سیستم لایک/دیسلایک - توابع جدید
    /**
     * لایک کردن کاربر
     */
    public function likeUser($user_id, $liked_user_id) {
        $user = $this->getUser($liked_user_id);
        if (!$user) return false;
        
        $current_likes = $user['likes'] ?? 0;
        $this->updateUser($liked_user_id, [
            'likes' => $current_likes + 1
        ]);
        
        if (!isset($this->data['likes_history'][$user_id])) {
            $this->data['likes_history'][$user_id] = [];
        }
        
        if (!in_array($liked_user_id, $this->data['likes_history'][$user_id])) {
            $this->data['likes_history'][$user_id][] = $liked_user_id;
            $this->saveData();
            return true;
        }
        
        return false;
    }
    
    /**
     * دیسلایک کردن کاربر
     */
    public function dislikeUser($user_id, $disliked_user_id) {
        $user = $this->getUser($disliked_user_id);
        if (!$user) return false;
        
        $current_dislikes = $user['dislikes'] ?? 0;
        $this->updateUser($disliked_user_id, [
            'dislikes' => $current_dislikes + 1
        ]);
        
        if (!isset($this->data['dislikes_history'][$user_id])) {
            $this->data['dislikes_history'][$user_id] = [];
        }
        
        if (!in_array($disliked_user_id, $this->data['dislikes_history'][$user_id])) {
            $this->data['dislikes_history'][$user_id][] = $disliked_user_id;
            $this->saveData();
            return true;
        }
        
        return false;
    }
    
    /**
     * بررسی آیا کاربر قبلاً لایک کرده؟
     */
    public function hasLiked($user_id, $target_user_id) {
        return isset($this->data['likes_history'][$user_id]) && 
               in_array($target_user_id, $this->data['likes_history'][$user_id]);
    }
    
    /**
     * بررسی آیا کاربر قبلاً دیسلایک کرده؟
     */
    public function hasDisliked($user_id, $target_user_id) {
        return isset($this->data['dislikes_history'][$user_id]) && 
               in_array($target_user_id, $this->data['dislikes_history'][$user_id]);
    }
    
    /**
     * دریافت امتیاز کاربر (لایک‌ها منهای دیسلایک‌ها)
     */
    public function getUserRating($user_id) {
        $user = $this->getUser($user_id);
        if (!$user) return 0;
        
        $likes = $user['likes'] ?? 0;
        $dislikes = $user['dislikes'] ?? 0;
        
        return $likes - $dislikes;
    }
    
    /**
     * دریافت برترین کاربران بر اساس امتیاز
     */
    public function getTopUsers($limit = 10) {
        $users_with_rating = [];
        
        foreach ($this->data['users'] as $user) {
            $rating = $this->getUserRating($user['user_id']);
            $users_with_rating[] = [
                'user_id' => $user['user_id'],
                'display_name' => $user['display_name'] ?? 'کاربر',
                'rating' => $rating,
                'likes' => $user['likes'] ?? 0,
                'dislikes' => $user['dislikes'] ?? 0
            ];
        }
        
        usort($users_with_rating, function($a, $b) {
            return $b['rating'] - $a['rating'];
        });
        
        return array_slice($users_with_rating, 0, $limit);
    }
}
?>