<?php
// handlers/messagehandler.php
require_once __DIR__ . '/../core/SupportManager.php';
require_once __DIR__ . '/../core/AnonLinkManager.php';

class MessageHandler {
    private $telegram;
    private $userManager;
    private $supportManager;
    private $anonLinkManager;
    private $db;
    
    public function __construct($telegram, $userManager, $db_connection = null) {
        $this->telegram = $telegram;
        $this->userManager = $userManager;
        $this->supportManager = new SupportManager();
        
        // اتصال به دیتابیس برای AnonLinkManager
        if ($db_connection) {
            $this->db = $db_connection;
            $this->anonLinkManager = new AnonLinkManager($db_connection);
        }
    }
    
    public function handle($update) {
        if (isset($update['callback_query'])) {
            $this->handleCallback($update);
            return;
        }
        
        $message = $update['message'] ?? null;
        if (!$message) return;
        
        $chat_id = $message['chat']['id'];
        $user_id = $message['from']['id'];
        $text = $message['text'] ?? '';
        $first_name = $message['from']['first_name'] ?? '';
        $last_name = $message['from']['last_name'] ?? '';
        $username = $message['from']['username'] ?? '';
        
        $this->userManager->createUser($user_id, [
            'first_name' => $first_name,
            'last_name' => $last_name,
            'username' => $username
        ]);
        
        // هندل لینک ناشناس
        if (strpos($text, '/start anon_') === 0) {
            $this->handleAnonLinkStart($chat_id, $user_id, $text);
            return;
        }
        
        if ($this->userManager->isBlocked($user_id)) {
            $this->telegram->sendMessage($chat_id, "🚫 حساب شما مسدود شده است. لطفا با پشتیبانی تماس بگیرید.");
            return;
        }
        
        if (strpos($text, '/start invite_') === 0) {
            $this->handleInviteLink($chat_id, $user_id, $text);
            return;
        }
        
        if ($text === '/start') {
            $this->handleStart($chat_id, $user_id, $message);
            return;
        }
        
        if (strpos($text, '/support') === 0 && $this->isAdmin($user_id)) {
            $this->showAdminSupport($chat_id, $user_id);
            return;
        }
        
        if (strpos($text, '/reply_') === 0 && $this->isAdmin($user_id)) {
            $this->handleAdminReply($chat_id, $user_id, $text);
            return;
        }
        
        if (strpos($text, '/toptens') === 0 && $this->isAdmin($user_id)) {
            $this->showTopUsersAdmin($chat_id, $user_id);
            return;
        }
        
        $this->processCommand($chat_id, $user_id, $text, $message);
    }
    
    private function handleAnonLinkStart($chat_id, $user_id, $text) {
        // استخراج کد لینک
        $link_code = str_replace('/start anon_', '', $text);
        
        if (!$this->anonLinkManager) {
            $this->telegram->sendMessage($chat_id, "⚠️ سیستم لینک ناشناس در حال حاضر غیرفعال است.");
            return;
        }
        
        // دریافت اطلاعات لینک
        $link = $this->anonLinkManager->getLinkByCode($link_code);
        
        if (!$link) {
            $this->telegram->sendMessage($chat_id, 
                "⚠️ این لینک معتبر نیست یا منقضی شده است.\n" .
                "لینک ممکن است:\n" .
                "• توسط سازنده حذف شده باشد\n" .
                "• تاریخ انقضایش گذشته باشد\n" .
                "• غیرفعال شده باشد"
            );
            return;
        }
        
        // ذخیره اطلاعات در سشن کاربر
        $this->userManager->updateUser($user_id, [
            'anon_link_code' => $link_code,
            'step' => 'anonlink_message_input'
        ]);
        
        // نمایش پیام خوشآمد
        $owner_name = $link['owner_name'] ?? 'یک کاربر';
        $welcome_text = "👁‍🗨 *ارسال پیام ناشناس*\n\n";
        $welcome_text .= "شما در حال ارسال پیام ناشناس به *{$owner_name}* هستید.\n\n";
        $welcome_text .= "📝 *نکات مهم:*\n";
        $welcome_text .= "• پیام شما کاملاً ناشناس می‌ماند\n";
        $welcome_text .= "• دریافت‌کننده نمی‌داند شما کیستید\n";
        $welcome_text .= "• پس از ارسال امکان ویرایش یا حذف ندارید\n\n";
        $welcome_text .= "لطفا پیام خود را بنویسید:";
        
        $this->telegram->sendMessage($chat_id, $welcome_text);
    }
    
    private function handleInviteLink($chat_id, $user_id, $text) {
        $invite_code = str_replace('/start invite_', '', $text);
        $invite_result = $this->userManager->handleInvite($user_id, $invite_code);
        
        if ($invite_result['success']) {
            $inviter_id = $invite_result['inviter_id'];
            
            $this->telegram->sendMessage($chat_id, "🎉 ۱۰ سکه هدیه برای عضویت از طریق دعوت دریافت کردید!");
            
            $invited_user = $this->userManager->getUser($user_id);
            $invited_name = $invited_user['first_name'] . ($invited_user['last_name'] ? ' ' . $invited_user['last_name'] : '');
            
            $congrats_message = "🎊 *تبریک!*\n\n";
            $congrats_message .= "کاربر {$invited_name} با لینک دعوت شما وارد ربات شد.\n";
            $congrats_message .= "🎁 ۱۰ سکه به حساب شما اضافه شد!\n";
            $congrats_message .= "💰 سکه‌های شما: " . $this->userManager->getCoins($inviter_id) . "\n";
            $congrats_message .= "👥 تعداد دعوت‌ها: " . ($this->userManager->getUser($inviter_id)['invited_count'] ?? 0);
            
            $this->telegram->sendMessage($inviter_id, $congrats_message);
        } else {
            $this->telegram->sendMessage($chat_id, "⚠️ لینک دعوت نامعتبر یا منقضی شده است.");
        }
        
        $user = $this->userManager->getUser($user_id);
        if (empty($user['display_name'])) {
            $this->showNameInput($chat_id, $user_id);
        } else {
            $this->showMainMenu($chat_id, $user_id);
        }
    }
    
    private function handleStart($chat_id, $user_id, $message) {
        $user = $this->userManager->getUser($user_id);
        
        if (empty($user['display_name'])) {
            $this->showNameInput($chat_id, $user_id);
        } else {
            $this->showMainMenu($chat_id, $user_id);
        }
    }
    
    private function showNameInput($chat_id, $user_id) {
        $user = $this->userManager->getUser($user_id);
        $current_name = $user['display_name'] ?? '';
        
        $text = "👋 سلام! به ربات چت ناشناس خوش آمدید!\n\n";
        
        if ($current_name) {
            $text .= "نام فعلی شما: *{$current_name}*\n";
            $text .= "اگر می‌خواهید نام خود را تغییر دهید، نام جدید را ارسال کنید.\n";
            $text .= "در غیر این صورت، همین نام را تأیید کنید:";
            
            $keyboard = [
                [['text' => '✅ تأیید همین نام']],
                [['text' => '✏️ تغییر نام']]
            ];
        } else {
            $text .= "لطفا نام نمایشی خود را وارد کنید:\n";
            $text .= "(این نام در پروفایل شما نمایش داده می‌شود)";
            
            $keyboard = [];
        }
        
        $this->telegram->sendKeyboard($chat_id, $text, $keyboard);
        $this->userManager->updateUser($user_id, ['step' => 'set_name']);
    }
    
    private function checkChannelMembership($user_id) {
        $test_result = $this->telegram->getChatMember('@jobaychat', $user_id);
        
        if ($test_result === false) return true;
        
        if ($test_result && $test_result['ok']) {
            $status = $test_result['result']['status'];
            return in_array($status, ['member', 'administrator', 'creator']);
        }
        
        return true;
    }
    
    private function showChannelRequirement($chat_id) {
        $text = "📢 *برای استفاده از ربات، باید در کانال ما عضو شوید*\n\n";
        $text .= "لطفا در کانال زیر عضو شوید و سپس /start را بفرستید:";
        
        $keyboard = [
            [['text' => "📍 عضویت در کانال", 'url' => "https://t.me/jobaychat"]],
            [['text' => "✅ بررسی عضویت", 'callback_data' => 'check_membership']]
        ];
        
        $this->telegram->sendInlineKeyboard($chat_id, $text, $keyboard);
    }
    
    private function processCommand($chat_id, $user_id, $text, $message) {
        $user = $this->userManager->getUser($user_id);
        
        $partner_id = $this->userManager->getChatPartner($user_id);
        if ($partner_id) {
            $this->handleChatCommands($chat_id, $user_id, $text);
            return;
        }
        
        if (in_array($text, ['/start', '💬 شروع چت ناشناس', '🎲 چت رندوم (رایگان)', '👨 چت با پسر (۲ سکه)', '👩 چت با دختر (۲ سکه)'])) {
            if (!$this->checkChannelMembership($user_id)) {
                $this->showChannelRequirement($chat_id);
                return;
            }
        }
        
        switch ($text) {
            case '/start':
                $this->handleStart($chat_id, $user_id, $message);
                break;
                
            case '/menu':
            case 'منو':
            case '🔙 بازگشت به منوی اصلی':
                $this->showMainMenu($chat_id, $user_id);
                break;
                
            case '💬 شروع چت ناشناس':
                $this->showChatTypeMenu($chat_id, $user_id);
                break;
                
            case '🎲 چت رندوم (رایگان)':
                $this->handleRandomChat($chat_id, $user_id);
                break;
                
            case '👨 چت با پسر (۲ سکه)':
                $this->handleFilteredChat($chat_id, $user_id, 'پسر');
                break;
                
            case '👩 چت با دختر (۲ سکه)':
                $this->handleFilteredChat($chat_id, $user_id, 'دختر');
                break;
                
            case '🔙 بازگشت به انواع چت':
                $this->showChatTypeMenu($chat_id, $user_id);
                break;
                
            case '👤 پروفایل من':
                $this->showProfile($chat_id, $user_id);
                break;
                
            case '✏️ ویرایش پروفایل':
                $this->showEditProfile($chat_id, $user_id);
                break;
                
            case '📞 پشتیبانی':
                $this->showSupportMenu($chat_id, $user_id);
                break;
                
            case '📨 ارسال پیام به پشتیبانی':
                $this->askForSupportMessage($chat_id, $user_id);
                break;
                
            case '📋 وضعیت تیکت‌های من':
                $this->showMyTickets($chat_id, $user_id);
                break;
                
            case '❓ سوالات متداول':
                $this->showFAQ($chat_id);
                break;
                
            case '🎁 لینک ناشناس': // 🔥 جدید
                $this->showAnonLinkMenu($chat_id, $user_id);
                break;
                
            case '💰 فروشگاه':
                $this->showCoinShop($chat_id, $user_id);
                break;
                
            case '🔙 بازگشت':
                $this->showProfile($chat_id, $user_id);
                break;
                
            case '🔙 بازگشت به پشتیبانی':
                $this->showSupportMenu($chat_id, $user_id);
                break;
                
            case '🔙 انصراف از جستجو':
                $this->cancelSearch($chat_id, $user_id);
                break;
                
            case '👥 دعوت دوستان':
                $this->showInvitePanel($chat_id, $user_id);
                break;
                
            case '✅ تأیید همین نام':
                $this->handleConfirmName($chat_id, $user_id);
                break;
                
            case '✏️ تغییر نام':
                $this->handleChangeName($chat_id, $user_id);
                break;
                
            case '📨 ارسال تیکت جدید':
                $this->askForSupportMessage($chat_id, $user_id);
                break;
                
            case '🏆 برترین کاربران':
                $this->showTopUsers($chat_id);
                break;
                
            // 🎁 منوی لینک ناشناس
            case '🔗 ساخت لینک جدید':
                $this->handleCreateAnonLink($chat_id, $user_id);
                break;
                
            case '📨 پیام‌های دریافتی':
                $this->showAnonLinkMessages($chat_id, $user_id);
                break;
                
            case '📊 آمار و مدیریت':
                $this->showAnonLinkStats($chat_id, $user_id);
                break;
                
            case '🔙 بازگشت به منوی لینک':
                $this->showAnonLinkMenu($chat_id, $user_id);
                break;
                
            // 🏪 منوی فروشگاه جدید
            case '🪙 خرید سکه':
                $this->showBuyCoinsMenu($chat_id, $user_id);
                break;
                
            case '⭐️ حساب ویژه':
                $this->showVIPInfo($chat_id, $user_id);
                break;
                
            case '🔙 بازگشت به فروشگاه':
                $this->showCoinShop($chat_id, $user_id);
                break;
                
            case '❌ انصراف از ارسال':
                $this->cancelAnonMessage($chat_id, $user_id);
                break;
                
            default:
                $this->handleDefault($chat_id, $user_id, $text, $user);
                break;
        }
    }
    
    private function handleChatCommands($chat_id, $user_id, $text) {
        $system_commands = [
            '✂️ پایان چت', '⭐️ نفر بعدی', '🛑 گزارش کاربر', 
            '🚫 بلاک کاربر', '👍 لایک', '👎 دیسلایک'
        ];
        
        if (in_array($text, $system_commands)) {
            switch ($text) {
                case '✂️ پایان چت':
                    $this->endChat($chat_id, $user_id);
                    break;
                    
                case '⭐️ نفر بعدی':
                    $this->nextUser($chat_id, $user_id);
                    break;
                    
                case '🛑 گزارش کاربر':
                    $this->reportUser($chat_id, $user_id);
                    break;
                    
                case '🚫 بلاک کاربر':
                    $this->blockPartner($chat_id, $user_id);
                    break;
                    
                case '👍 لایک':
                    $this->handleLike($chat_id, $user_id);
                    break;
                    
                case '👎 دیسلایک':
                    $this->handleDislike($chat_id, $user_id);
                    break;
            }
        } else {
            $this->forwardMessageToPartner($chat_id, $user_id, $text);
        }
    }
    
    private function showChatMenu($chat_id) {
        $keyboard = [
            [['text' => '👎 دیسلایک'], ['text' => '👍 لایک']],
            [['text' => '✂️ پایان چت'], ['text' => '⭐️ نفر بعدی']],
            [['text' => '🚫 بلاک کاربر'], ['text' => '🛑 گزارش کاربر']]
        ];
        
        $text = "💬 *شما در حال چت هستید*\n\n";
        $text .= "برای مدیریت چت از دکمه‌های زیر استفاده کنید:";
        
        $this->telegram->sendKeyboard($chat_id, $text, $keyboard);
    }
    
    private function handleLike($chat_id, $user_id) {
        $partner_id = $this->userManager->getChatPartner($user_id);
        
        if (!$partner_id) {
            $this->telegram->sendMessage($chat_id, "❌ ارتباط قطع شده است.");
            $this->showMainMenu($chat_id, $user_id);
            return;
        }
        
        if ($this->userManager->hasLiked($user_id, $partner_id)) {
            $this->telegram->sendMessage($chat_id, "✅ شما قبلاً به این کاربر لایک داده‌اید!");
            return;
        }
        
        $this->userManager->likeUser($user_id, $partner_id);
        
        $this->telegram->sendMessage($chat_id, 
            "👍 *لایک شما ثبت شد!*\n\n" .
            "امتیاز شما برای کاربر مقابل ذخیره شد.\n" .
            "از چت با این کاربر لذت بردید؟ می‌توانید ادامه دهید یا نفر بعدی را امتحان کنید."
        );
        
        $partner = $this->userManager->getUser($partner_id);
        $current_likes = $partner['likes'] ?? 0;
        
        $notification = "🎉 *کسی به شما لایک داد!*\n\n";
        $notification .= "یکی از کاربران از چت با شما لذت برد.\n";
        $notification .= "📊 تعداد لایک‌های شما: {$current_likes}";
        
        $this->telegram->sendMessage($partner_id, $notification);
    }
    
    private function handleDislike($chat_id, $user_id) {
        $partner_id = $this->userManager->getChatPartner($user_id);
        
        if (!$partner_id) {
            $this->telegram->sendMessage($chat_id, "❌ ارتباط قطع شده است.");
            $this->showMainMenu($chat_id, $user_id);
            return;
        }
        
        if ($this->userManager->hasDisliked($user_id, $partner_id)) {
            $this->telegram->sendMessage($chat_id, "👎 شما قبلاً به این کاربر دیسلایک داده‌اید!");
            return;
        }
        
        $this->userManager->dislikeUser($user_id, $partner_id);
        
        $this->telegram->sendMessage($chat_id, 
            "👎 *دیسلایک شما ثبت شد.*\n\n" .
            "بازخورد شما برای بهبود سیستم ذخیره شد.\n" .
            "می‌خواهید چت را ادامه دهید یا نفر بعدی را امتحان کنید؟"
        );
        
        $this->showChatMenu($chat_id);
    }
    
    private function forwardMessageToPartner($chat_id, $user_id, $text) {
        $partner_id = $this->userManager->getChatPartner($user_id);
        
        if ($partner_id) {
            $this->telegram->sendMessage($partner_id, $text);
        } else {
            $this->telegram->sendMessage($chat_id, "❌ ارتباط قطع شده است.");
            $this->showMainMenu($chat_id, $user_id);
        }
    }
    
    private function showGenderSelection($chat_id, $user_id) {
        $keyboard = [
            [['text' => '👩 دختر'], ['text' => '👨 پسر']]
        ];
        
        $user = $this->userManager->getUser($user_id);
        $display_name = $user['display_name'] ?? 'کاربر';
        
        $text = "👋 سلام {$display_name}!\n\n";
        $text .= "لطفا جنسیت خود را انتخاب کنید:";
        
        $this->telegram->sendKeyboard($chat_id, $text, $keyboard);
        $this->userManager->updateUser($user_id, ['step' => 'set_gender']);
    }
    
    // 🔥 تغییر اصلی: منوی اصلی جدید
    private function showMainMenu($chat_id, $user_id) {
        $user = $this->userManager->getUser($user_id);
        $vip_status = $user['vip'] ? "⭐️ ویژه" : "🔹 عادی";
        $coins = $this->userManager->getCoins($user_id);
        $display_name = $user['display_name'] ?? 'کاربر';
        $rating = $this->userManager->getUserRating($user_id);
        $rating_emoji = $rating > 0 ? '📈' : ($rating < 0 ? '📉' : '📊');
        
        $text = "🎯 *منوی اصلی - {$display_name}*\n\n";
        $text .= "👤 وضعیت حساب: {$vip_status}\n";
        $text .= "💰 سکه‌ها: {$coins}\n";
        $text .= "{$rating_emoji} امتیاز: {$rating}\n";
        $text .= "💬 تعداد چت‌ها: " . ($user['chat_count'] ?? 0) . "\n\n";
        $text .= "لطفا انتخاب کنید:";
        
        $keyboard = [
            [['text' => '💬 شروع چت ناشناس']],
            [['text' => '👤 پروفایل من'], ['text' => '🎁 لینک ناشناس']],
            [['text' => '👥 دعوت دوستان'], ['text' => '💰 فروشگاه']],
            [['text' => '📞 پشتیبانی'], ['text' => '🏆 برترین کاربران']]
        ];
        
        $this->telegram->sendKeyboard($chat_id, $text, $keyboard);
        $this->userManager->updateUser($user_id, ['step' => 'main_menu']);
    }
    
    private function showChatTypeMenu($chat_id, $user_id) {
        $user = $this->userManager->getUser($user_id);
        $coins = $this->userManager->getCoins($user_id);
        
        $text = "💬 *انتخاب نوع چت*\n\n";
        $text .= "💰 سکه‌های شما: {$coins}\n\n";
        $text .= "لطفا نوع چت مورد نظر را انتخاب کنید:";
        
        $keyboard = [
            [['text' => '🎲 چت رندوم (رایگان)']],
            [['text' => '👨 چت با پسر (۲ سکه)'], ['text' => '👩 چت با دختر (۲ سکه)']],
            [['text' => '🔙 بازگشت به منوی اصلی']]
        ];
        
        $this->telegram->sendKeyboard($chat_id, $text, $keyboard);
        $this->userManager->updateUser($user_id, ['step' => 'chat_type_menu']);
    }
    
    private function showProfile($chat_id, $user_id) {
        $user = $this->userManager->getUser($user_id);
        $vip_status = $user['vip'] ? "⭐️ ویژه" : "🔹 عادی";
        $sex_emoji = ($user['sex'] == 'دختر') ? '👩' : '👨';
        $rating = $this->userManager->getUserRating($user_id);
        $rating_emoji = $rating > 0 ? '📈' : ($rating < 0 ? '📉' : '📊');
        
        $text = "👤 *پروفایل شما*\n\n";
        $text .= "👤 نام: " . ($user['display_name'] ?? 'تعیین نشده') . "\n";
        $text .= "{$sex_emoji} جنسیت: " . ($user['sex'] ?? 'تعیین نشده') . "\n";
        $text .= "🎂 سن: " . ($user['age'] ?? 'تعیین نشده') . " سال\n";
        $text .= "🏙 شهر: " . ($user['city'] ?? 'تعیین نشده') . "\n";
        $text .= "⭐️ وضعیت: {$vip_status}\n";
        $text .= "💰 سکه‌ها: " . ($this->userManager->getCoins($user_id) ?? 0) . "\n";
        $text .= "💬 تعداد چت‌ها: " . ($user['chat_count'] ?? 0) . "\n";
        $text .= "{$rating_emoji} امتیاز: {$rating} | 👍 {$user['likes']} | 👎 {$user['dislikes']}\n";
        $text .= "👥 دعوت شده‌ها: " . ($user['invited_count'] ?? 0) . "\n\n";
        
        $keyboard = [
            [['text' => '✏️ ویرایش پروفایل']],
            [['text' => '🔙 بازگشت به منوی اصلی']]
        ];
        
        $this->telegram->sendKeyboard($chat_id, $text, $keyboard);
    }
    
    private function showTopUsers($chat_id) {
        $top_users = $this->userManager->getTopUsers(10);
        
        $text = "🏆 *برترین کاربران*\n\n";
        $text .= "رتبه‌بندی بر اساس امتیاز:\n\n";
        
        foreach ($top_users as $index => $user) {
            $medal = match($index) {
                0 => "🥇",
                1 => "🥈", 
                2 => "🥉",
                default => ($index + 1) . "."
            };
            
            $text .= "{$medal} {$user['display_name']}\n";
            $text .= "   ⭐ امتیاز: {$user['rating']} | 👍 {$user['likes']} | 👎 {$user['dislikes']}\n\n";
        }
        
        $keyboard = [
            [['text' => '👤 پروفایل من']],
            [['text' => '🔙 بازگشت به منوی اصلی']]
        ];
        
        $this->telegram->sendKeyboard($chat_id, $text, $keyboard);
    }
    
    private function showEditProfile($chat_id, $user_id) {
        $text = "✏️ *ویرایش پروفایل*\n\n";
        $text .= "لطفا بخش مورد نظر برای ویرایش را انتخاب کنید:";
        
        $keyboard = [
            [['text' => '👤 تغییر نام']],
            [['text' => '👩/👨 تغییر جنسیت']],
            [['text' => '🎂 تغییر سن']],
            [['text' => '🏙 تغییر شهر']],
            [['text' => '🔙 بازگشت']]
        ];
        
        $this->telegram->sendKeyboard($chat_id, $text, $keyboard);
        $this->userManager->updateUser($user_id, ['step' => 'edit_profile_menu']);
    }
    
    private function showSupportMenu($chat_id, $user_id) {
        $text = "📞 *سیستم پشتیبانی*\n\n";
        $text .= "لطفا گزینه مورد نظر را انتخاب کنید:";
        
        $keyboard = [
            [['text' => '📨 ارسال پیام به پشتیبانی']],
            [['text' => '📋 وضعیت تیکت‌های من']],
            [['text' => '❓ سوالات متداول']],
            [['text' => '🔙 بازگشت به منوی اصلی']]
        ];
        
        $this->telegram->sendKeyboard($chat_id, $text, $keyboard);
        $this->userManager->updateUser($user_id, ['step' => 'support_menu']);
    }
    
    private function askForSupportMessage($chat_id, $user_id) {
        $text = "📝 *پیام خود را بنویسید:*\n\n";
        $text .= "لطفا مشکل، پیشنهاد یا سوال خود را به طور کامل توضیح دهید.\n";
        $text .= "پاسخ شما در اسرع وقت ارسال خواهد شد.\n\n";
        $text .= "⚠️ نکته: پیام باید حداقل ۱۰ کاراکتر باشد.";
        
        $this->telegram->sendMessage($chat_id, $text);
        $this->userManager->updateUser($user_id, ['step' => 'support_message']);
    }
    
    private function showMyTickets($chat_id, $user_id) {
        $tickets = $this->supportManager->getUserTickets($user_id, 3);
        
        if (empty($tickets)) {
            $text = "📋 *تیکت‌های شما*\n\n";
            $text .= "شما تاکنون هیچ تیکتی ثبت نکرده‌اید.\n\n";
            $text .= "برای ثبت تیکت جدید، گزینه '📨 ارسال پیام به پشتیبانی' را انتخاب کنید.";
        } else {
            $text = "📋 *آخرین تیکت‌های شما*\n\n";
            
            foreach ($tickets as $ticket) {
                $status_emoji = ($ticket['status'] == 'open') ? '🟡' : '🟢';
                $status_text = ($ticket['status'] == 'open') ? 'در انتظار پاسخ' : 'پاسخ داده شده';
                
                $text .= "{$status_emoji} *{$ticket['ticket_code']}*\n";
                $text .= "📝 " . substr($ticket['message'], 0, 50) . "...\n";
                $text .= "📅 " . $ticket['created_at'] . "\n";
                $text .= "🔸 وضعیت: {$status_text}\n\n";
            }
            
            $text .= "برای مشاهده جزئیات بیشتر، کد تیکت را به ادمین ارسال کنید.";
        }
        
        $keyboard = [
            [['text' => '📨 ارسال تیکت جدید']],
            [['text' => '🔙 بازگشت به پشتیبانی']]
        ];
        
        $this->telegram->sendKeyboard($chat_id, $text, $keyboard);
    }
    
    private function showFAQ($chat_id) {
        $faqs = [
            [
                'question' => 'چگونه سکه خریداری کنم؟',
                'answer' => 'از منوی اصلی > فروشگاه > خرید سکه می‌توانید سکه خریداری کنید.'
            ],
            [
                'question' => 'چت فیلتر شده چیست؟',
                'answer' => 'امکان چت با جنسیت خاص با هزینه ۲ سکه. کاربران VIP رایگان هستند.'
            ],
            [
                'question' => 'چگونه دوستان را دعوت کنم؟',
                'answer' => 'از بخش دعوت دوستان، لینک خود را ارسال کنید. برای هر دعوت ۱۰ سکه جایزه بگیرید.'
            ],
            [
                'question' => 'لایک و دیسلایک چیست؟',
                'answer' => 'امتیازدهی به کاربران. لایک برای چت خوب، دیسلایک برای رفتار نامناسب.'
            ],
            [
                'question' => 'لینک ناشناس چیست؟',
                'answer' => 'لینک اختصاصی برای دریافت پیام ناشناس از دیگران. هزینه: ۱۰ سکه'
            ]
        ];
        
        $text = "❓ *سوالات متداول*\n\n";
        
        foreach ($faqs as $index => $faq) {
            $text .= ($index + 1) . ". *{$faq['question']}*\n";
            $text .= "{$faq['answer']}\n\n";
        }
        
        $text .= "اگر پاسخ خود را نیافتید، از بخش پشتیبانی سوال خود را بپرسید.";
        
        $keyboard = [
            [['text' => '📨 ارسال سوال جدید']],
            [['text' => '🔙 بازگشت']]
        ];
        
        $this->telegram->sendKeyboard($chat_id, $text, $keyboard);
    }
    
    // 🔥 تغییر: انتقال VIP به فروشگاه
    private function showVIPInfo($chat_id, $user_id) {
        $user = $this->userManager->getUser($user_id);
        
        $text = "⭐️ *حساب ویژه*\n\n";
        
        if ($user['vip']) {
            $text .= "✅ حساب شما ویژه است!\n\n";
            $text .= "امکانات فعال:\n";
            $text .= "• ✅ چت نامحدود\n";
            $text .= "• ✅ ارسال عکس و فایل\n";
            $text .= "• ✅ مشاهده اطلاعات کاربران\n";
            $text .= "• ✅ اولویت در اتصال\n";
        } else {
            $text .= "🔹 حساب شما عادی است\n\n";
            $text .= "🆓 *روش رایگان:*\n";
            $text .= "۳ دوستان را دعوت کنید\n\n";
            $text .= "💳 *روش پرداخت:*\n";
            $text .= "پرداخت ۵,۰۰۰ تومان\n\n";
            $text .= "امکانات حساب ویژه:\n";
            $text .= "• 🔸 چت نامحدود\n";
            $text .= "• 🔸 ارسال عکس و فایل\n";
            $text .= "• 🔸 مشاهده اطلاعات کاربران\n";
            $text .= "• 🔸 اولویت در اتصال\n";
            
            $inline_keyboard = [
                [
                    ['text' => '👥 دعوت دوستان', 'callback_data' => 'vip_invite'],
                    ['text' => '💳 پرداخت', 'callback_data' => 'vip_pay']
                ]
            ];
            
            $this->telegram->sendInlineKeyboard($chat_id, $text, $inline_keyboard);
            return;
        }
        
        $keyboard = [
            [['text' => '🔙 بازگشت به فروشگاه']]
        ];
        
        $this->telegram->sendKeyboard($chat_id, $text, $keyboard);
    }
    
    private function showPartnerProfile($chat_id, $partner_id) {
        $partner = $this->userManager->getUser($partner_id);
        
        if ($partner) {
            $sex_emoji = ($partner['sex'] == 'دختر') ? '👩' : '👨';
            $partner_rating = $this->userManager->getUserRating($partner_id);
            $partner_rating_emoji = $partner_rating > 0 ? '📈' : ($partner_rating < 0 ? '📉' : '📊');
            
            $text = "👤 *پروفایل کاربر مقابل*\n\n";
            $text .= "👤 نام: " . ($partner['display_name'] ?? 'نامشخص') . "\n";
            $text .= "{$sex_emoji} جنسیت: " . ($partner['sex'] ?? 'نامشخص') . "\n";
            $text .= "🎂 سن: " . ($partner['age'] ?? 'نامشخص') . " سال\n";
            $text .= "🏙 شهر: " . ($partner['city'] ?? 'نامشخص') . "\n";
            $text .= "💬 تعداد چت‌ها: " . ($partner['chat_count'] ?? 0) . "\n";
            $text .= "{$partner_rating_emoji} امتیاز: {$partner_rating}\n";
            $text .= "👍 لایک: " . ($partner['likes'] ?? 0) . " | 👎 دیسلایک: " . ($partner['dislikes'] ?? 0) . "\n\n";
            $text .= "📝 می‌توانید چت را شروع کنید...";
            
            $this->telegram->sendMessage($chat_id, $text);
        }
    }
    
    private function handleDefault($chat_id, $user_id, $text, $user) {
        $step = $user['step'] ?? 'none';
        
        switch ($step) {
            case 'set_name':
                $this->handleNameInput($chat_id, $user_id, $text);
                break;
                
            case 'set_gender':
                $this->handleGenderSelection($chat_id, $user_id, $text);
                break;
                
            case 'set_age':
                $this->handleAgeSelection($chat_id, $user_id, $text);
                break;
                
            case 'set_city':
                $this->handleCitySelection($chat_id, $user_id, $text);
                break;
                
            case 'chat_type_menu':
                $this->handleChatTypeMenu($chat_id, $user_id, $text);
                break;
                
            case 'edit_profile_menu':
                $this->handleEditProfileMenu($chat_id, $user_id, $text);
                break;
                
            case 'edit_name':
                $this->handleEditName($chat_id, $user_id, $text);
                break;
                
            case 'edit_gender':
                $this->handleEditGender($chat_id, $user_id, $text);
                break;
                
            case 'edit_age':
                $this->handleEditAge($chat_id, $user_id, $text);
                break;
                
            case 'edit_city':
                $this->handleEditCity($chat_id, $user_id, $text);
                break;
                
            case 'support_message':
                $this->handleSupportMessage($chat_id, $user_id, $text);
                break;
                
            // 🎁 مراحل جدید لینک ناشناس
            case 'anonlink_message_input':
                $this->handleAnonMessageInput($chat_id, $user_id, $text);
                break;
                
            case 'anonlink_confirm':
                $this->handleAnonMessageConfirm($chat_id, $user_id, $text);
                break;
                
            case 'anonlink_create_name':
                $this->handleAnonLinkName($chat_id, $user_id, $text);
                break;
                
            default:
                $this->telegram->sendMessage($chat_id, "🤔 دستور نامعتبر است. لطفا از منو استفاده کنید.");
                $this->showMainMenu($chat_id, $user_id);
                break;
        }
    }
    
    private function handleChatTypeMenu($chat_id, $user_id, $text) {
        switch ($text) {
            case '🎲 چت رندوم (رایگان)':
                $this->handleRandomChat($chat_id, $user_id);
                break;
                
            case '👨 چت با پسر (۲ سکه)':
                $this->handleFilteredChat($chat_id, $user_id, 'پسر');
                break;
                
            case '👩 چت با دختر (۲ سکه)':
                $this->handleFilteredChat($chat_id, $user_id, 'دختر');
                break;
                
            case '🔙 بازگشت به منوی اصلی':
                $this->showMainMenu($chat_id, $user_id);
                break;
                
            default:
                $this->telegram->sendMessage($chat_id, "لطفا از دکمه‌های زیر استفاده کنید.");
                $this->showChatTypeMenu($chat_id, $user_id);
                break;
        }
    }
    
    private function handleNameInput($chat_id, $user_id, $text) {
        if (strlen(trim($text)) < 2) {
            $this->telegram->sendMessage($chat_id, "❌ نام باید حداقل ۲ حرف باشد. لطفا نام معتبری وارد کنید:");
            return;
        }
        
        if (strlen(trim($text)) > 30) {
            $this->telegram->sendMessage($chat_id, "❌ نام نمی‌تواند بیشتر از ۳۰ حرف باشد. لطفا نام کوتاه‌تری وارد کنید:");
            return;
        }
        
        $this->userManager->updateUser($user_id, [
            'display_name' => trim($text),
            'step' => 'set_gender'
        ]);
        
        $this->showGenderSelection($chat_id, $user_id);
    }
    
    private function handleGenderSelection($chat_id, $user_id, $text) {
        if ($text === '👩 دختر' || $text === '👨 پسر') {
            $gender = ($text === '👩 دختر') ? 'دختر' : 'پسر';
            
            $this->userManager->updateUser($user_id, [
                'sex' => $gender,
                'step' => 'set_age'
            ]);
            
            $this->showAgeSelection($chat_id, $user_id);
        } else {
            $this->telegram->sendMessage($chat_id, "لطفا جنسیت خود را از دکمه‌ها انتخاب کنید.");
        }
    }
    
    private function showAgeSelection($chat_id, $user_id) {
        $keyboard = [
            [['text' => '13'], ['text' => '14'], ['text' => '15'], ['text' => '16']],
            [['text' => '17'], ['text' => '18'], ['text' => '19'], ['text' => '20']],
            [['text' => '21'], ['text' => '22'], ['text' => '23'], ['text' => '24']],
            [['text' => '25'], ['text' => '26'], ['text' => '27'], ['text' => '28']],
            [['text' => '29'], ['text' => '30'], ['text' => '31'], ['text' => '32']],
            [['text' => '32+']]
        ];
        
        $text = "✅ جنسیت شما ثبت شد!\n\n";
        $text .= "🎂 لطفا سن خود را انتخاب کنید:";
        
        $this->telegram->sendKeyboard($chat_id, $text, $keyboard);
    }
    
    private function handleAgeSelection($chat_id, $user_id, $text) {
        if (is_numeric($text) || $text === '32+') {
            $age = $text;
            
            $this->userManager->updateUser($user_id, [
                'age' => $age,
                'step' => 'set_city'
            ]);
            
            $this->showCitySelection($chat_id, $user_id);
        } else {
            $this->telegram->sendMessage($chat_id, "لطفا سن خود را از دکمه‌ها انتخاب کنید.");
        }
    }
    
    private function showCitySelection($chat_id, $user_id) {
        $keyboard = [
            [['text' => 'تهران'], ['text' => 'مشهد'], ['text' => 'اصفهان']],
            [['text' => 'شیراز'], ['text' => 'تبریز'], ['text' => 'کرج']],
            [['text' => 'اهواز'], ['text' => 'قم'], ['text' => 'کرمانشاه']],
            [['text' => 'ارومیه'], ['text' => 'رشت'], ['text' => 'زاهدان']],
            [['text' => 'همدان'], ['text' => 'یزد'], ['text' => 'اردبیل']],
            [['text' => 'سایر']]
        ];
        
        $text = "✅ سن شما ثبت شد!\n\n";
        $text .= "🏙 لطفا شهر خود را انتخاب کنید:";
        
        $this->telegram->sendKeyboard($chat_id, $text, $keyboard);
    }
    
    private function handleCitySelection($chat_id, $user_id, $text) {
        $this->userManager->updateUser($user_id, [
            'city' => $text,
            'step' => 'main_menu'
        ]);
        
        $this->userManager->giveWelcomeCoins($user_id);
        
        $text = "🎉 *ثبت نام کامل شد!*\n\n";
        $text .= "✅ اطلاعات شما با موفقیت ثبت شد.\n";
        $text .= "🎁 ۵ سکه هدیه به شما تعلق گرفت!\n";
        $text .= "حالا می‌توانید از چت ناشناس استفاده کنید.";
        
        $this->showMainMenu($chat_id, $user_id);
    }
    
    private function handleEditProfileMenu($chat_id, $user_id, $text) {
        switch ($text) {
            case '👤 تغییر نام':
                $this->telegram->sendMessage($chat_id, "لطفا نام جدید خود را وارد کنید:");
                $this->userManager->updateUser($user_id, ['step' => 'edit_name']);
                break;
                
            case '👩/👨 تغییر جنسیت':
                $this->showEditGender($chat_id, $user_id);
                break;
                
            case '🎂 تغییر سن':
                $this->showEditAge($chat_id, $user_id);
                break;
                
            case '🏙 تغییر شهر':
                $this->showEditCity($chat_id, $user_id);
                break;
                
            case '🔙 بازگشت':
                $this->showProfile($chat_id, $user_id);
                break;
                
            default:
                $this->telegram->sendMessage($chat_id, "لطفا از دکمه‌های زیر استفاده کنید.");
                $this->showEditProfile($chat_id, $user_id);
                break;
        }
    }
    
    private function handleEditName($chat_id, $user_id, $text) {
        if (strlen(trim($text)) < 2) {
            $this->telegram->sendMessage($chat_id, "❌ نام باید حداقل ۲ حرف باشد. لطفا نام معتبری وارد کنید:");
            return;
        }
        
        if (strlen(trim($text)) > 30) {
            $this->telegram->sendMessage($chat_id, "❌ نام نمی‌تواند بیشتر از ۳۰ حرف باشد. لطفا نام کوتاه‌تری وارد کنید:");
            return;
        }
        
        $this->userManager->updateUser($user_id, [
            'display_name' => trim($text),
            'step' => 'main_menu'
        ]);
        
        $this->telegram->sendMessage($chat_id, "✅ نام شما با موفقیت تغییر کرد!");
        $this->showMainMenu($chat_id, $user_id);
    }
    
    private function showEditGender($chat_id, $user_id) {
        $keyboard = [
            [['text' => '👩 دختر'], ['text' => '👨 پسر']],
            [['text' => '🔙 بازگشت']]
        ];
        
        $text = "لطفا جنسیت جدید خود را انتخاب کنید:";
        
        $this->telegram->sendKeyboard($chat_id, $text, $keyboard);
        $this->userManager->updateUser($user_id, ['step' => 'edit_gender']);
    }
    
    private function handleEditGender($chat_id, $user_id, $text) {
        if ($text === '🔙 بازگشت') {
            $this->showEditProfile($chat_id, $user_id);
            return;
        }
        
        if ($text === '👩 دختر' || $text === '👨 پسر') {
            $gender = ($text === '👩 دختر') ? 'دختر' : 'پسر';
            
            $this->userManager->updateUser($user_id, [
                'sex' => $gender,
                'step' => 'main_menu'
            ]);
            
            $this->telegram->sendMessage($chat_id, "✅ جنسیت شما با موفقیت تغییر کرد!");
            $this->showMainMenu($chat_id, $user_id);
        } else {
            $this->telegram->sendMessage($chat_id, "لطفا جنسیت خود را از دکمه‌ها انتخاب کنید.");
        }
    }
    
    private function showEditAge($chat_id, $user_id) {
        $keyboard = [
            [['text' => '13'], ['text' => '14'], ['text' => '15'], ['text' => '16']],
            [['text' => '17'], ['text' => '18'], ['text' => '19'], ['text' => '20']],
            [['text' => '21'], ['text' => '22'], ['text' => '23'], ['text' => '24']],
            [['text' => '25'], ['text' => '26'], ['text' => '27'], ['text' => '28']],
            [['text' => '29'], ['text' => '30'], ['text' => '31'], ['text' => '32']],
            [['text' => '32+'], ['text' => '🔙 بازگشت']]
        ];
        
        $text = "لطفا سن جدید خود را انتخاب کنید:";
        
        $this->telegram->sendKeyboard($chat_id, $text, $keyboard);
        $this->userManager->updateUser($user_id, ['step' => 'edit_age']);
    }
    
    private function handleEditAge($chat_id, $user_id, $text) {
        if ($text === '🔙 بازگشت') {
            $this->showEditProfile($chat_id, $user_id);
            return;
        }
        
        if (is_numeric($text) || $text === '32+') {
            $this->userManager->updateUser($user_id, [
                'age' => $text,
                'step' => 'main_menu'
            ]);
            
            $this->telegram->sendMessage($chat_id, "✅ سن شما با موفقیت تغییر کرد!");
            $this->showMainMenu($chat_id, $user_id);
        } else {
            $this->telegram->sendMessage($chat_id, "لطفا سن خود را از دکمه‌ها انتخاب کنید.");
        }
    }
    
    private function showEditCity($chat_id, $user_id) {
        $keyboard = [
            [['text' => 'تهران'], ['text' => 'مشهد'], ['text' => 'اصفهان']],
            [['text' => 'شیراز'], ['text' => 'تبریز'], ['text' => 'کرج']],
            [['text' => 'اهواز'], ['text' => 'قم'], ['text' => 'کرمانشاه']],
            [['text' => 'ارومیه'], ['text' => 'رشت'], ['text' => 'زاهدان']],
            [['text' => 'همدان'], ['text' => 'یزد'], ['text' => 'اردبیل']],
            [['text' => 'سایر'], ['text' => '🔙 بازگشت']]
        ];
        
        $text = "لطفا شهر جدید خود را انتخاب کنید:";
        
        $this->telegram->sendKeyboard($chat_id, $text, $keyboard);
        $this->userManager->updateUser($user_id, ['step' => 'edit_city']);
    }
    
    private function handleEditCity($chat_id, $user_id, $text) {
        if ($text === '🔙 بازگشت') {
            $this->showEditProfile($chat_id, $user_id);
            return;
        }
        
        $this->userManager->updateUser($user_id, [
            'city' => $text,
            'step' => 'main_menu'
        ]);
        
        $this->telegram->sendMessage($chat_id, "✅ شهر شما با موفقیت تغییر کرد!");
        $this->showMainMenu($chat_id, $user_id);
    }
    
    private function handleSupportMessage($chat_id, $user_id, $text) {
        if (strlen(trim($text)) < 10) {
            $this->telegram->sendMessage($chat_id, 
                "❌ پیام شما خیلی کوتاه است!\n\n" .
                "لطفا توضیح کامل‌تری ارائه دهید (حداقل ۱۰ کاراکتر).\n" .
                "دوباره پیام خود را ارسال کنید:"
            );
            return;
        }
        
        $result = $this->supportManager->createTicket($user_id, trim($text));
        
        if ($result['success']) {
            $this->notifyAdmins($user_id, trim($text), $result['ticket_code']);
            
            $this->telegram->sendMessage($chat_id, 
                "✅ *پیام شما با موفقیت ثبت شد!*\n\n" .
                "🆔 کد تیکت: `{$result['ticket_code']}`\n" .
                "📅 تاریخ ثبت: " . date('Y-m-d H:i:s') . "\n\n" .
                "کارشناسان ما در اسرع وقت پاسخ خواهند داد.\n" .
                "می‌توانید وضعیت پیام خود را از بخش '📋 وضعیت تیکت‌های من' پیگیری کنید."
            );
            
            $this->showSupportMenu($chat_id, $user_id);
        } else {
            $this->telegram->sendMessage($chat_id, 
                "❌ خطا در ثبت پیام!\n\n" .
                "لطفا مجدداً تلاش کنید یا با ادمین تماس بگیرید."
            );
        }
    }
    
    private function handleConfirmName($chat_id, $user_id) {
        $user = $this->userManager->getUser($user_id);
        $current_name = $user['display_name'] ?? '';
        
        if ($current_name) {
            $this->showGenderSelection($chat_id, $user_id);
        } else {
            $this->telegram->sendMessage($chat_id, "لطفا ابتدا نام خود را وارد کنید.");
            $this->showNameInput($chat_id, $user_id);
        }
    }
    
    private function handleChangeName($chat_id, $user_id) {
        $this->telegram->sendMessage($chat_id, "لطفا نام جدید خود را ارسال کنید:");
        $this->userManager->updateUser($user_id, ['step' => 'set_name']);
    }
    
    private function handleRandomChat($chat_id, $user_id) {
        $this->handleChatRequest($chat_id, $user_id);
    }
    
    private function handleFilteredChat($chat_id, $user_id, $preferred_gender) {
        $user = $this->userManager->getUser($user_id);
        
        if (empty($user['sex']) || empty($user['age']) || empty($user['city']) || empty($user['display_name'])) {
            $this->telegram->sendMessage($chat_id, "لطفا ابتدا پروفایل خود را تکمیل کنید.");
            $this->showGenderSelection($chat_id, $user_id);
            return;
        }
        
        $user_gender = $user['sex'];
        if ($user_gender == $preferred_gender) {
            $this->telegram->sendMessage($chat_id, "⚠️ شما نمی‌توانید با جنسیت خودتان چت کنید.");
            $this->showMainMenu($chat_id, $user_id);
            return;
        }
        
        $result = $this->userManager->requestFilteredChat($user_id, $preferred_gender);
        
        if (!$result['success']) {
            if (isset($result['error']) && $result['error'] == 'کمبود سکه') {
                $coins_needed = $result['coins_needed'] ?? 2;
                $current_coins = $this->userManager->getCoins($user_id);
                
                $text = "❌ *موجودی سکه کافی نیست!*\n\n";
                $text .= "💰 سکه‌های شما: {$current_coins}\n";
                $text .= "💸 برای چت فیلتر شده به {$coins_needed} سکه نیاز دارید.\n\n";
                $text .= "🎲 می‌توانید از *چت رندوم رایگان* استفاده کنید.";
                
                $keyboard = [
                    [['text' => '🎲 چت رندوم (رایگان)']],
                    [['text' => '👥 دعوت دوستان'], ['text' => '💰 فروشگاه']],
                    [['text' => '🔙 بازگشت']]
                ];
                
                $this->telegram->sendKeyboard($chat_id, $text, $keyboard);
            }
            return;
        }
        
        $coins_deducted = $result['coins_deducted'] ?? 0;
        
        $this->telegram->sendMessage($chat_id, "🔍 در حال جستجوی {$preferred_gender}...");
        
        $partner_id = $this->userManager->findFilteredPartner($user_id);
        
        if ($partner_id) {
            $chat_id_result = $this->userManager->createChat($user_id, $partner_id);
            
            if ($chat_id_result) {
                $message = "✅ یک {$preferred_gender} پیدا شد!";
                if ($coins_deducted > 0) {
                    $message .= " ({$coins_deducted} سکه کسر شد)";
                }
                
                $this->telegram->sendMessage($chat_id, $message);
                $this->telegram->sendMessage($partner_id, "✅ یک کاربر پیدا شد!");
                
                $this->showPartnerProfile($chat_id, $partner_id);
                $this->showPartnerProfile($partner_id, $user_id);
                
                $this->showChatMenu($chat_id);
                $this->showChatMenu($partner_id);
            }
        } else {
            $message = "⏳ در صف انتظار برای {$preferred_gender}...\n\n";
            $message .= "💰 هزینه: ۲ سکه\n";
            $message .= "⏰ زمان انتظار: ۱-۵ دقیقه\n\n";
            $message .= "می‌توانید منتظر بمانید یا:";
            
            $keyboard = [
                [['text' => '🎲 چت رندوم (رایگان)']],
                [['text' => '🔙 انصراف از جستجو']]
            ];
            
            $this->telegram->sendKeyboard($chat_id, $message, $keyboard);
        }
    }
    
    private function cancelSearch($chat_id, $user_id) {
        $this->userManager->removeFromFilteredWaiting($user_id);
        $this->userManager->removeFromWaitingList($user_id);
        
        $this->telegram->sendMessage($chat_id, "✅ جستجو متوقف شد.");
        $this->showMainMenu($chat_id, $user_id);
    }
    
    private function handleChatRequest($chat_id, $user_id) {
        $user = $this->userManager->getUser($user_id);
        
        if (empty($user['sex']) || empty($user['age']) || empty($user['city']) || empty($user['display_name'])) {
            $this->telegram->sendMessage($chat_id, "لطفا ابتدا پروفایل خود را تکمیل کنید.");
            $this->showNameInput($chat_id, $user_id);
            return;
        }
        
        $this->telegram->sendMessage($chat_id, "🔍 در حال جستجوی کاربر تصادفی...");
        
        $this->userManager->addToWaitingList($user_id);
        
        $waiting_users = $this->userManager->getWaitingList();
        $other_users = array_filter($waiting_users, function($id) use ($user_id) {
            return $id != $user_id;
        });
        
        if (count($other_users) > 0) {
            $random_user = $other_users[array_rand($other_users)];
            $chat_id_result = $this->userManager->createChat($user_id, $random_user);
            
            if ($chat_id_result) {
                $this->telegram->sendMessage($chat_id, "✅ کاربری پیدا شد! (رایگان)");
                $this->telegram->sendMessage($random_user, "✅ کاربری پیدا شد!");
                
                $this->showPartnerProfile($chat_id, $random_user);
                $this->showPartnerProfile($random_user, $user_id);
                
                $this->showChatMenu($chat_id);
                $this->showChatMenu($random_user);
            }
        } else {
            $this->telegram->sendMessage($chat_id, "⏳ در صف انتظار...");
        }
    }
    
    private function endChat($chat_id, $user_id) {
        $partner_id = $this->userManager->getChatPartner($user_id);
        
        if ($partner_id) {
            $user_liked = $this->userManager->hasLiked($user_id, $partner_id);
            $partner_liked = $this->userManager->hasLiked($partner_id, $user_id);
            
            $this->userManager->endChat($user_id);
            
            $user = $this->userManager->getUser($user_id);
            
            if (isset($user['last_refund']) && isset($user['refund_amount'])) {
                $refund_amount = $user['refund_amount'];
                $current_coins = $this->userManager->getCoins($user_id);
                
                $message = "✅ چت به پایان رسید.\n\n";
                $message .= "⏰ چت کمتر از ۱۰ ثانیه بود!\n";
                $message .= "💎 {$refund_amount} سکه به حساب شما بازگشت.\n";
                $message .= "💰 موجودی فعلی: {$current_coins} سکه";
                
                $this->telegram->sendMessage($chat_id, $message);
                
                $this->userManager->updateUser($user_id, [
                    'last_refund' => null,
                    'refund_amount' => null
                ]);
            } else {
                $this->telegram->sendMessage($chat_id, "✅ چت به پایان رسید.");
            }
            
            if ($user_liked && $partner_liked) {
                $this->telegram->sendMessage($chat_id, "\n💕 عالی! هر دو به هم لایک دادید!");
                $this->telegram->sendMessage($partner_id, "\n💕 عالی! هر دو به هم لایک دادید!");
            }
            
            $partner_message = "❌ کاربر مقابل چت را پایان داد.";
            
            $partner = $this->userManager->getUser($partner_id);
            if (isset($partner['last_refund']) && isset($partner['refund_amount'])) {
                $partner_refund = $partner['refund_amount'];
                $partner_coins = $this->userManager->getCoins($partner_id);
                
                $partner_message .= "\n\n⏰ چت کمتر از ۱۰ ثانیه بود!\n";
                $partner_message .= "💎 {$partner_refund} سکه به حساب شما بازگشت.\n";
                $partner_message .= "💰 موجودی فعلی: {$partner_coins} سکه";
                
                $this->userManager->updateUser($partner_id, [
                    'last_refund' => null,
                    'refund_amount' => null
                ]);
            }
            
            $this->telegram->sendMessage($partner_id, $partner_message);
            
            $this->showMainMenu($chat_id, $user_id);
            $this->showMainMenu($partner_id, $partner_id);
        } else {
            $this->telegram->sendMessage($chat_id, "❌ ارتباط قبلاً قطع شده است.");
            $this->showMainMenu($chat_id, $user_id);
        }
    }
    
    private function nextUser($chat_id, $user_id) {
        $this->endChat($chat_id, $user_id);
        sleep(1);
        $this->handleChatRequest($chat_id, $user_id);
    }
    
    private function reportUser($chat_id, $user_id) {
        $partner_id = $this->userManager->getChatPartner($user_id);
        
        if ($partner_id) {
            $this->telegram->sendMessage($chat_id, "✅ گزارش شما ارسال شد. از همکاری شما متشکریم.");
        }
    }
    
    private function blockPartner($chat_id, $user_id) {
        $partner_id = $this->userManager->getChatPartner($user_id);
        
        if ($partner_id) {
            $this->userManager->blockUser($partner_id);
            $this->telegram->sendMessage($chat_id, "✅ کاربر مقابل بلاک شد.");
            $this->endChat($chat_id, $user_id);
        }
    }
    
    private function showInvitePanel($chat_id, $user_id) {
        $invite_link = $this->userManager->generateInviteLink($user_id);
        $user = $this->userManager->getUser($user_id);
        $coins = $this->userManager->getCoins($user_id);
        
        $text = "👥 *دعوت از دوستان*\n\n";
        $text .= "💰 سکه‌های شما: *{$coins}*\n";
        $text .= "👥 دعوت شده‌ها: *{$user['invited_count']}*\n\n";
        $text .= "🎁 *پاداش‌ها:*\n";
        $text .= "• ۱۰ سکه برای هر دعوت\n";
        $text .= "• ۱۰ سکه به کاربر جدید\n";
        $text .= "• 📨 اطلاع‌رسانی هنگام عضویت دوستان\n\n";
        $text .= "🔗 لینک دعوت شما:\n`{$invite_link}`";
        
        $keyboard = [
            [['text' => '💰 فروشگاه']],
            [['text' => '🔙 بازگشت به منوی اصلی']]
        ];
        
        $this->telegram->sendKeyboard($chat_id, $text, $keyboard);
    }
    
    // 🔥 تغییر: منوی فروشگاه جدید
    private function showCoinShop($chat_id, $user_id) {
        $coins = $this->userManager->getCoins($user_id);
        
        $text = "🏪 *فروشگاه*\n\n";
        $text .= "💰 موجودی: *{$coins}* سکه\n\n";
        $text .= "لطفا انتخاب کنید:";
        
        $keyboard = [
            [['text' => '🪙 خرید سکه'], ['text' => '⭐️ حساب ویژه']],
            [['text' => '🎁 سایر محصولات']],
            [['text' => '🔙 بازگشت به منوی اصلی']]
        ];
        
        $this->telegram->sendKeyboard($chat_id, $text, $keyboard);
        $this->userManager->updateUser($user_id, ['step' => 'shop_menu']);
    }
    
    // 🎁 منوی خرید سکه جدید
    private function showBuyCoinsMenu($chat_id, $user_id) {
        $coins = $this->userManager->getCoins($user_id);
        
        $text = "🪙 *خرید سکه*\n\n";
        $text .= "💰 موجودی: *{$coins}* سکه\n\n";
        $text .= "💎 *پکیج‌های سکه:*\n\n";
        $text .= "• ۱۰۰ سکه - ۱۰,۰۰۰ تومان\n";
        $text .= "• ۵۰۰ سکه - ۴۵,۰۰۰ تومان (۱۰٪ تخفیف)\n";
        $text .= "• ۱۰۰۰ سکه - ۸۰,۰۰۰ تومان (۲۰٪ تخفیف)\n";
        $text .= "• ۲۰۰۰ سکه - ۱۵۰,۰۰۰ تومان (۲۵٪ تخفیف)\n\n";
        $text .= "📞 برای خرید با ادمین تماس بگیرید: @admin";
        
        $keyboard = [
            [['text' => '🔙 بازگشت به فروشگاه']]
        ];
        
        $this->telegram->sendKeyboard($chat_id, $text, $keyboard);
    }
    
    // 🎁 سیستم لینک ناشناس
    private function showAnonLinkMenu($chat_id, $user_id) {
        $user = $this->userManager->getUser($user_id);
        
        if (!$this->anonLinkManager) {
            $this->telegram->sendMessage($chat_id, "⚠️ سیستم لینک ناشناس در حال حاضر غیرفعال است.");
            $this->showMainMenu($chat_id, $user_id);
            return;
        }
        
        $active_links = $this->anonLinkManager->getUserLinks($user_id, true);
        $total_messages = 0;
        
        foreach ($active_links as $link) {
            $total_messages += $link['message_count'];
        }
        
        $text = "🎁 *سیستم لینک ناشناس*\n\n";
        $text .= "با این سیستم می‌توانید:\n";
        $text .= "• لینک اختصاصی دریافت کنید\n";
        $text .= "• در کانال/بیو قرار دهید\n";
        $text .= "• پیام‌های ناشناس بگیرید\n\n";
        $text .= "📊 آمار شما:\n";
        $text .= "🔗 لینک‌های فعال: " . count($active_links) . "\n";
        $text .= "📨 کل پیام‌ها: {$total_messages}\n";
        $text .= "💰 هزینه ساخت: ۱۰ سکه\n";
        $text .= "⏰ مدت اعتبار: ۳۰ روز\n\n";
        $text .= "لطفا انتخاب کنید:";
        
        $keyboard = [
            [['text' => '🔗 ساخت لینک جدید']],
            [['text' => '📨 پیام‌های دریافتی']],
            [['text' => '📊 آمار و مدیریت']],
            [['text' => '🔙 بازگشت به منوی اصلی']]
        ];
        
        $this->telegram->sendKeyboard($chat_id, $text, $keyboard);
        $this->userManager->updateUser($user_id, ['step' => 'anonlink_menu']);
    }
    
    private function handleCreateAnonLink($chat_id, $user_id) {
        $user = $this->userManager->getUser($user_id);
        $coins = $this->userManager->getCoins($user_id);
        
        if ($coins < 10) {
            $this->telegram->sendMessage($chat_id,
                "❌ *موجودی سکه کافی نیست!*\n\n" .
                "💰 سکه‌های شما: {$coins}\n" .
                "💸 برای ساخت لینک ناشناس به ۱۰ سکه نیاز دارید.\n\n" .
                "می‌توانید:\n" .
                "• از بخش دعوت دوستان سکه کسب کنید\n" .
                "• از فروشگاه سکه بخرید"
            );
            
            $keyboard = [
                [['text' => '👥 دعوت دوستان']],
                [['text' => '💰 فروشگاه']],
                [['text' => '🔙 بازگشت']]
            ];
            
            $this->telegram->sendKeyboard($chat_id, "لطفا انتخاب کنید:", $keyboard);
            return;
        }
        
        $this->telegram->sendMessage($chat_id,
            "🔗 *ساخت لینک ناشناس*\n\n" .
            "لطفا یک نام برای لینک خود انتخاب کنید:\n" .
            "(مثال: 'نظرات درباره کانال' یا 'انتقادات مخفی')\n\n" .
            "⚠️ نکته: این نام فقط برای شما نمایش داده می‌شود."
        );
        
        $this->userManager->updateUser($user_id, ['step' => 'anonlink_create_name']);
    }
    
    private function handleAnonLinkName($chat_id, $user_id, $text) {
        if (strlen(trim($text)) < 2) {
            $this->telegram->sendMessage($chat_id, "❌ نام لینک باید حداقل ۲ حرف باشد.");
            return;
        }
        
        if (strlen(trim($text)) > 100) {
            $this->telegram->sendMessage($chat_id, "❌ نام لینک نمی‌تواند بیشتر از ۱۰۰ حرف باشد.");
            return;
        }
        
        // ساخت لینک
        $result = $this->anonLinkManager->createLink($user_id, trim($text));
        
        if ($result['success']) {
            $this->telegram->sendMessage($chat_id,
                "✅ *لینک ناشناس ساخته شد!*\n\n" .
                "📝 نام لینک: {$text}\n" .
                "🔗 لینک شما:\n`{$result['telegram_link']}`\n\n" .
                "⏰ اعتبار: تا " . date('Y-m-d', strtotime($result['expires_at'])) . "\n" .
                "💰 هزینه: {$result['coins_deducted']} سکه\n\n" .
                "📋 *نحوه استفاده:*\n" .
                "۱. این لینک را در کانال/بیو قرار دهید\n" .
                "۲. کاربران روی آن کلیک می‌کنند\n" .
                "۳. پیام ناشناس ارسال می‌کنند\n" .
                "۴. شما پیام را دریافت می‌کنید\n\n" .
                "⚠️ هشدار: پیام‌ها کاملاً ناشناس هستند."
            );
            
            $this->showAnonLinkMenu($chat_id, $user_id);
        } else {
            $this->telegram->sendMessage($chat_id,
                "❌ خطا در ساخت لینک!\n" .
                "خطا: " . ($result['error'] ?? 'ناشناخته')
            );
        }
    }
    
    private function showAnonLinkMessages($chat_id, $user_id) {
        if (!$this->anonLinkManager) {
            $this->telegram->sendMessage($chat_id, "⚠️ سیستم لینک ناشناس در حال حاضر غیرفعال است.");
            return;
        }
        
        $links = $this->anonLinkManager->getUserLinks($user_id, true);
        
        if (empty($links)) {
            $this->telegram->sendMessage($chat_id,
                "📨 *پیام‌های دریافتی*\n\n" .
                "شما هنوز هیچ لینک ناشناس فعالی ندارید.\n" .
                "برای ساخت لینک، گزینه '🔗 ساخت لینک جدید' را انتخاب کنید."
            );
            return;
        }
        
        $text = "📨 *پیام‌های دریافتی*\n\n";
        $text .= "لیست لینک‌های شما:\n\n";
        
        foreach ($links as $link) {
            $messages = $this->anonLinkManager->getLinkMessages($link['id'], 3);
            
            $text .= "🔗 *{$link['link_name']}*\n";
            $text .= "📊 پیام‌ها: {$link['message_count']}\n";
            $text .= "⏰ آخرین پیام: " . ($link['last_message_at'] ? date('Y-m-d H:i', strtotime($link['last_message_at'])) : 'هنوز پیامی ندارید') . "\n";
            
            if (!empty($messages)) {
                $text .= "📝 آخرین پیام‌ها:\n";
                foreach ($messages as $msg) {
                    $text .= "• " . substr($msg['message'], 0, 30) . "...\n";
                }
            }
            
            $text .= "\n";
        }
        
        $keyboard = [
            [['text' => '🔙 بازگشت به منوی لینک']]
        ];
        
        $this->telegram->sendKeyboard($chat_id, $text, $keyboard);
    }
    
    private function showAnonLinkStats($chat_id, $user_id) {
        if (!$this->anonLinkManager) {
            $this->telegram->sendMessage($chat_id, "⚠️ سیستم لینک ناشناس در حال حاضر غیرفعال است.");
            return;
        }
        
        $links = $this->anonLinkManager->getUserLinks($user_id, false);
        
        if (empty($links)) {
            $this->telegram->sendMessage($chat_id,
                "📊 *آمار و مدیریت*\n\n" .
                "شما هنوز هیچ لینکی نساخته‌اید."
            );
            return;
        }
        
        $text = "📊 *آمار و مدیریت لینک‌ها*\n\n";
        
        foreach ($links as $index => $link) {
            $stats = $this->anonLinkManager->getLinkStats($link['id']);
            $status = $link['is_active'] ? '🟢 فعال' : '🔴 غیرفعال';
            
            $text .= ($index + 1) . ". *{$link['link_name']}*\n";
            $text .= "   🔗 کد: `{$link['link_code']}`\n";
            $text .= "   📊 وضعیت: {$status}\n";
            $text .= "   📨 کل پیام‌ها: " . ($stats['total_messages'] ?? 0) . "\n";
            $text .= "   👁 خوانده شده: " . ($stats['read_messages'] ?? 0) . "\n";
            $text .= "   📅 ایجاد: " . date('Y-m-d', strtotime($link['created_at'])) . "\n";
            
            if ($link['expires_at']) {
                $text .= "   ⏰ انقضا: " . date('Y-m-d', strtotime($link['expires_at'])) . "\n";
            }
            
            $text .= "\n";
        }
        
        $keyboard = [
            [['text' => '🔙 بازگشت به منوی لینک']]
        ];
        
        $this->telegram->sendKeyboard($chat_id, $text, $keyboard);
    }
    
    private function handleAnonMessageInput($chat_id, $user_id, $text) {
        $user = $this->userManager->getUser($user_id);
        $link_code = $user['anon_link_code'] ?? '';
        
        if (empty($link_code)) {
            $this->telegram->sendMessage($chat_id, "❌ خطا در پردازش لینک.");
            $this->showMainMenu($chat_id, $user_id);
            return;
        }
        
        if (strlen(trim($text)) < 2) {
            $this->telegram->sendMessage($chat_id, "❌ پیام باید حداقل ۲ حرف باشد.");
            return;
        }
        
        if (strlen(trim($text)) > 1000) {
            $this->telegram->sendMessage($chat_id, "❌ پیام نمی‌تواند بیشتر از ۱۰۰۰ حرف باشد.");
            return;
        }
        
        // ذخیره پیام موقت
        $this->userManager->updateUser($user_id, [
            'temp_anon_message' => $text,
            'step' => 'anonlink_confirm'
        ]);
        
        $confirm_text = "👁‍🗨 *تأیید ارسال پیام ناشناس*\n\n";
        $confirm_text .= "📝 پیام شما:\n";
        $confirm_text .= "« " . $text . " »\n\n";
        $confirm_text .= "⚠️ *توجه:*\n";
        $confirm_text .= "• پیام شما کاملاً ناشناس ارسال می‌شود\n";
        $confirm_text .= "• امکان ویرایش یا حذف بعدی وجود ندارد\n";
        $confirm_text .= "• دریافت‌کننده نمی‌داند شما کیستید\n\n";
        $confirm_text .= "آیا مطمئنید می‌خواهید ارسال کنید؟";
        
        $keyboard = [
            [['text' => '✅ بله، ارسال کن']],
            [['text' => '❌ خیر، ویرایش کن']],
            [['text' => '❌ انصراف از ارسال']]
        ];
        
        $this->telegram->sendKeyboard($chat_id, $confirm_text, $keyboard);
    }
    
    private function handleAnonMessageConfirm($chat_id, $user_id, $text) {
        $user = $this->userManager->getUser($user_id);
        $link_code = $user['anon_link_code'] ?? '';
        $message = $user['temp_anon_message'] ?? '';
        
        if ($text === '✅ بله، ارسال کن') {
            // ارسال پیام
            $result = $this->anonLinkManager->saveAnonymousMessage(
                $link_code, 
                $message,
                'text',
                $_SERVER['REMOTE_ADDR'] ?? null
            );
            
            if ($result['success']) {
                // اطلاع به صاحب لینک
                $owner_id = $result['link_owner_id'];
                $link = $this->anonLinkManager->getLinkByCode($link_code);
                
                $notification = "📬 *پیام ناشناس جدید!*\n\n";
                $notification .= "📝 پیام:\n";
                $notification .= "« " . substr($message, 0, 200) . (strlen($message) > 200 ? "..." : "") . " »\n\n";
                $notification .= "🔗 از لینک: {$link['link_name']}\n";
                $notification .= "⏰ زمان: " . date('Y-m-d H:i:s');
                
                $this->telegram->sendMessage($owner_id, $notification);
                
                // پاسخ به فرستنده
                $this->telegram->sendMessage($chat_id,
                    "✅ *پیام شما با موفقیت ارسال شد!*\n\n" .
                    "پیام شما به صورت کاملاً ناشناس ارسال شد.\n" .
                    "صاحب لینک نمی‌داند شما کیستید.\n\n" .
                    "💡 می‌توانید پیام دیگری ارسال کنید یا ربات را ترک کنید."
                );
            } else {
                $this->telegram->sendMessage($chat_id,
                    "❌ خطا در ارسال پیام!\n" .
                    "خطا: " . ($result['error'] ?? 'ناشناخته')
                );
            }
            
            // پاک کردن اطلاعات موقت
            $this->userManager->updateUser($user_id, [
                'anon_link_code' => null,
                'temp_anon_message' => null,
                'step' => 'main_menu'
            ]);
            
        } elseif ($text === '❌ خیر، ویرایش کن') {
            $this->telegram->sendMessage($chat_id, "لطفا پیام جدید خود را ارسال کنید:");
            $this->userManager->updateUser($user_id, ['step' => 'anonlink_message_input']);
            
        } elseif ($text === '❌ انصراف از ارسال') {
            $this->telegram->sendMessage($chat_id, "❌ ارسال پیام لغو شد.");
            $this->userManager->updateUser($user_id, [
                'anon_link_code' => null,
                'temp_anon_message' => null,
                'step' => 'main_menu'
            ]);
        }
    }
    
    private function cancelAnonMessage($chat_id, $user_id) {
        $this->userManager->updateUser($user_id, [
            'anon_link_code' => null,
            'temp_anon_message' => null,
            'step' => 'main_menu'
        ]);
        
        $this->telegram->sendMessage($chat_id, "❌ ارسال پیام لغو شد.");
        $this->showMainMenu($chat_id, $user_id);
    }
    
    private function isAdmin($user_id) {
        if (defined('ADMIN_IDS')) {
            $admin_ids = ADMIN_IDS;
        } else {
            $admin_ids = [123456789];
        }
        return in_array($user_id, $admin_ids);
    }
    
    private function showAdminSupport($chat_id, $user_id) {
        if (!$this->isAdmin($user_id)) {
            $this->telegram->sendMessage($chat_id, "❌ دسترسی denied!");
            return;
        }
        
        $stats = $this->supportManager->getStats();
        $open_tickets = $this->supportManager->getOpenTickets();
        
        $text = "👨‍💼 *پنل ادمین پشتیبانی*\n\n";
        $text .= "📊 آمار کلی:\n";
        $text .= "• 📨 کل تیکت‌ها: {$stats['total_tickets']}\n";
        $text .= "• 🟡 در انتظار: {$stats['pending_tickets']}\n";
        $text .= "• 🟢 پاسخ داده: {$stats['answered_tickets']}\n\n";
        
        if (!empty($open_tickets)) {
            $text .= "📋 تیکت‌های باز:\n";
            foreach ($open_tickets as $ticket) {
                $text .= "• `{$ticket['ticket_code']}` - کاربر: {$ticket['user_id']}\n";
                $text .= "  \"" . substr($ticket['message'], 0, 30) . "...\"\n";
            }
            $text .= "\nبرای پاسخ: /reply_کدتیکت";
        } else {
            $text .= "✅ هیچ تیکت بازی وجود ندارد.";
        }
        
        $this->telegram->sendMessage($chat_id, $text);
    }
    
    private function handleAdminReply($chat_id, $user_id, $text) {
        if (!$this->isAdmin($user_id)) {
            $this->telegram->sendMessage($chat_id, "❌ دسترسی denied!");
            return;
        }
        
        $parts = explode(' ', $text, 2);
        $ticket_code = str_replace('/reply_', '', $parts[0]);
        $response = $parts[1] ?? '';
        
        if (empty($response)) {
            $this->telegram->sendMessage($chat_id, 
                "❌ فرمت دستور اشتباه است!\n\n" .
                "فرمت صحیح:\n" .
                "/reply_کدتیکت پاسخ شما\n\n" .
                "مثال:\n" .
                "/reply_T20241115001 مشکل شما حل شد."
            );
            return;
        }
        
        $data = json_decode(file_get_contents(dirname(__DIR__) . '/data/support_tickets.json'), true);
        
        foreach ($data['tickets'] as &$ticket) {
            if ($ticket['ticket_code'] === $ticket_code) {
                $ticket['status'] = 'answered';
                $ticket['admin_id'] = $user_id;
                $ticket['response'] = $response;
                $ticket['answered_at'] = date('Y-m-d H:i:s');
                
                $user_message = "📬 *پاسخ پشتیبانی*\n\n";
                $user_message .= "🆔 کد تیکت: {$ticket_code}\n";
                $user_message .= "📝 پاسخ: {$response}\n\n";
                $user_message .= "✅ مشکل شما حل شد؟\n";
                $user_message .= "اگر نیاز به پاسخ بیشتر دارید، دوباره پیام بفرستید.";
                
                $this->telegram->sendMessage($ticket['user_id'], $user_message);
                
                $data['statistics']['pending_tickets']--;
                $data['statistics']['answered_tickets']++;
                
                file_put_contents(dirname(__DIR__) . '/data/support_tickets.json', 
                    json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
                
                $this->telegram->sendMessage($chat_id, "✅ پاسخ با موفقیت ارسال شد!");
                return;
            }
        }
        
        $this->telegram->sendMessage($chat_id, "❌ تیکتی با این کد یافت نشد!");
    }
    
    private function showTopUsersAdmin($chat_id, $user_id) {
        if (!$this->isAdmin($user_id)) {
            $this->telegram->sendMessage($chat_id, "❌ دسترسی denied!");
            return;
        }
        
        $top_users = $this->userManager->getTopUsers(20);
        
        $text = "🏆 *برترین کاربران (پنل ادمین)*\n\n";
        
        foreach ($top_users as $index => $user) {
            $text .= ($index + 1) . ". {$user['display_name']} (ID: {$user['user_id']})\n";
            $text .= "   ⭐ امتیاز: {$user['rating']} | 👍 {$user['likes']} | 👎 {$user['dislikes']}\n\n";
        }
        
        $this->telegram->sendMessage($chat_id, $text);
    }
    
    private function notifyAdmins($user_id, $message, $ticket_code) {
        $user = $this->userManager->getUser($user_id);
        $user_name = $user['display_name'] ?? "کاربر #{$user_id}";
        $user_info = ($user['first_name'] ?? '') . ' ' . ($user['last_name'] ?? '');
        
        $admin_ids = defined('ADMIN_IDS') ? ADMIN_IDS : [123456789];
        
        foreach ($admin_ids as $admin_id) {
            $text = "🔔 *تیکت پشتیبانی جدید*\n\n";
            $text .= "🆔 کد تیکت: `{$ticket_code}`\n";
            $text .= "👤 کاربر: {$user_name}\n";
            if (!empty(trim($user_info))) {
                $text .= "📞 اطلاعات: {$user_info}\n";
            }
            $text .= "🆔 آیدی: `{$user_id}`\n";
            $text .= "📝 پیام: " . substr($message, 0, 100) . "...\n\n";
            $text .= "⏰ زمان: " . date('Y-m-d H:i:s') . "\n\n";
            $text .= "📋 برای پاسخ: /reply_{$ticket_code}";
            
            $this->telegram->sendMessage($admin_id, $text);
        }
    }
    
    public function handleCallback($update) {
        $callback_query = $update['callback_query'];
        $data = $callback_query['data'];
        $chat_id = $callback_query['message']['chat']['id'];
        $user_id = $callback_query['from']['id'];
        $message_id = $callback_query['message']['message_id'];
        
        if ($data === 'check_membership') {
            if ($this->checkChannelMembership($user_id)) {
                $this->telegram->sendMessage($chat_id, "✅ عالی! شما در کانال عضو هستید. لطفا /start را بفرستید.");
                $this->telegram->deleteMessage($chat_id, $message_id);
            } else {
                $this->telegram->sendMessage($chat_id, "❌ هنوز در کانال عضو نشده‌اید. لطفا ابتدا عضو شوید.");
            }
        }
        
        $this->telegram->answerCallbackQuery($callback_query['id'], "در حال پردازش...");
    }
}
?>