<?php
// core/SimpleAnonLinkManager.php
class SimpleAnonLinkManager {
    private $dataFile;
    
    public function __construct() {
        $this->dataFile = dirname(__DIR__) . '/data/anon_links.json';
        $this->initData();
    }
    
    private function initData() {
        if (!file_exists($this->dataFile)) {
            $data = [
                'links' => [],
                'messages' => [],
                'statistics' => [
                    'total_links' => 0,
                    'total_messages' => 0,
                    'active_links' => 0
                ]
            ];
            file_put_contents($this->dataFile, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
        }
    }
    
    private function loadData() {
        $json = file_get_contents($this->dataFile);
        return json_decode($json, true);
    }
    
    private function saveData($data) {
        file_put_contents($this->dataFile, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    }
    
    // 🔗 ایجاد لینک جدید
    public function createLink($user_id, $link_name = 'لینک ناشناس') {
        $data = $this->loadData();
        
        // تولید کد منحصر به فرد
        $link_code = 'AL' . date('Ymd') . '_' . substr(md5($user_id . time()), 0, 8);
        
        $link_id = uniqid('link_', true);
        
        $link = [
            'id' => $link_id,
            'user_id' => $user_id,
            'link_code' => $link_code,
            'link_name' => $link_name,
            'created_at' => date('Y-m-d H:i:s'),
            'expires_at' => date('Y-m-d H:i:s', strtotime('+30 days')),
            'is_active' => true,
            'message_count' => 0,
            'last_message_at' => null,
            'views' => 0
        ];
        
        $data['links'][$link_id] = $link;
        $data['statistics']['total_links']++;
        $data['statistics']['active_links']++;
        
        $this->saveData($data);
        
        // ساخت لینک تلگرام
        $telegram_link = "https://t.me/" . BOT_USERNAME . "?start=anon_{$link_code}";
        
        return [
            'success' => true,
            'link_id' => $link_id,
            'link_code' => $link_code,
            'telegram_link' => $telegram_link,
            'expires_at' => $link['expires_at']
        ];
    }
    
    // 🔍 یافتن لینک با کد
    public function getLinkByCode($link_code) {
        $data = $this->loadData();
        
        foreach ($data['links'] as $link) {
            if ($link['link_code'] === $link_code && 
                $link['is_active'] && 
                strtotime($link['expires_at']) > time()) {
                
                // افزایش تعداد بازدید
                $data['links'][$link['id']]['views']++;
                $this->saveData($data);
                
                return $link;
            }
        }
        
        return null;
    }
    
    // 📨 ذخیره پیام ناشناس
    public function saveAnonymousMessage($link_code, $message, $sender_ip = null) {
        $data = $this->loadData();
        $link_found = null;
        $link_key = null;
        
        // یافتن لینک
        foreach ($data['links'] as $key => &$link) {
            if ($link['link_code'] === $link_code && 
                $link['is_active'] && 
                strtotime($link['expires_at']) > time()) {
                
                $link_found = &$link;
                $link_key = $key;
                break;
            }
        }
        
        if (!$link_found) {
            return ['success' => false, 'error' => 'لینک نامعتبر یا منقضی شده'];
        }
        
        // اعتبارسنجی پیام
        $message = trim($message);
        if (strlen($message) < 2) {
            return ['success' => false, 'error' => 'پیام خیلی کوتاه است'];
        }
        
        if (strlen($message) > 1000) {
            return ['success' => false, 'error' => 'پیام خیلی طولانی است'];
        }
        
        // ذخیره پیام
        $message_id = uniqid('msg_', true);
        
        $data['messages'][$message_id] = [
            'id' => $message_id,
            'link_id' => $link_found['id'],
            'message' => $message,
            'sender_ip' => $sender_ip,
            'created_at' => date('Y-m-d H:i:s'),
            'is_read' => false
        ];
        
        // آپدیت آمار لینک
        $link_found['message_count']++;
        $link_found['last_message_at'] = date('Y-m-d H:i:s');
        $data['links'][$link_key] = $link_found;
        
        $data['statistics']['total_messages']++;
        
        $this->saveData($data);
        
        return [
            'success' => true,
            'message_id' => $message_id,
            'link_owner_id' => $link_found['user_id'],
            'link_name' => $link_found['link_name']
        ];
    }
    
    // 👤 دریافت لینک‌های کاربر
    public function getUserLinks($user_id) {
        $data = $this->loadData();
        $user_links = [];
        
        foreach ($data['links'] as $link) {
            if ($link['user_id'] == $user_id && 
                $link['is_active'] && 
                strtotime($link['expires_at']) > time()) {
                $user_links[] = $link;
            }
        }
        
        return $user_links;
    }
    
    // 📩 دریافت پیام‌های لینک
    public function getLinkMessages($link_id, $limit = 20) {
        $data = $this->loadData();
        $link_messages = [];
        $count = 0;
        
        foreach ($data['messages'] as $message) {
            if ($message['link_id'] == $link_id) {
                $link_messages[] = $message;
                $count++;
                if ($count >= $limit) break;
            }
        }
        
        // مرتب سازی جدیدترین اول
        usort($link_messages, function($a, $b) {
            return strtotime($b['created_at']) - strtotime($a['created_at']);
        });
        
        return $link_messages;
    }
    
    // 📊 دریافت آمار
    public function getStats() {
        $data = $this->loadData();
        return $data['statistics'];
    }
    
    // 🗑️ حذف لینک
    public function deleteLink($link_id, $user_id) {
        $data = $this->loadData();
        
        if (isset($data['links'][$link_id]) && 
            $data['links'][$link_id]['user_id'] == $user_id) {
            
            $data['links'][$link_id]['is_active'] = false;
            $data['statistics']['active_links']--;
            
            $this->saveData($data);
            return true;
        }
        
        return false;
    }
    
    // 📈 دریافت آمار کلی کاربر
    public function getUserStats($user_id) {
        $links = $this->getUserLinks($user_id);
        $total_messages = 0;
        $total_views = 0;
        
        foreach ($links as $link) {
            $total_messages += $link['message_count'];
            $total_views += $link['views'];
        }
        
        return [
            'total_links' => count($links),
            'total_messages' => $total_messages,
            'total_views' => $total_views,
            'active_links' => count(array_filter($links, function($link) {
                return strtotime($link['expires_at']) > time();
            }))
        ];
    }
}
?>