<?php
// core/AnonLinkManager.php
class AnonLinkManager {
    private $db;
    
    public function __construct($db_connection) {
        $this->db = $db_connection;
        $this->createTables();
    }
    
    private function createTables() {
        $this->db->query("
            CREATE TABLE IF NOT EXISTS anon_links (
                id INT AUTO_INCREMENT PRIMARY KEY,
                user_id BIGINT NOT NULL,
                link_code VARCHAR(50) UNIQUE NOT NULL,
                link_name VARCHAR(100) DEFAULT 'لینک ناشناس',
                created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                expires_at DATETIME,
                is_active BOOLEAN DEFAULT TRUE,
                message_count INT DEFAULT 0,
                last_message_at DATETIME,
                settings TEXT,
                INDEX idx_user_id (user_id),
                INDEX idx_link_code (link_code)
            )
        ");
        
        $this->db->query("
            CREATE TABLE IF NOT EXISTS anon_messages (
                id INT AUTO_INCREMENT PRIMARY KEY,
                link_id INT NOT NULL,
                sender_ip VARCHAR(45),
                message TEXT NOT NULL,
                message_type ENUM('text', 'image', 'voice') DEFAULT 'text',
                is_read BOOLEAN DEFAULT FALSE,
                created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                deleted_at DATETIME,
                FOREIGN KEY (link_id) REFERENCES anon_links(id) ON DELETE CASCADE,
                INDEX idx_link_id (link_id),
                INDEX idx_created_at (created_at)
            )
        ");
    }
    
    /**
     * ایجاد لینک ناشناس جدید
     */
    public function createLink($user_id, $link_name = 'لینک ناشناس', $duration_days = 30) {
        // بررسی موجودی کاربر
        require_once __DIR__ . '/UserManager.php';
        $userManager = new UserManager();
        $user = $userManager->getUser($user_id);
        
        if ($user['coins'] < 1) {
            return ['success' => false, 'error' => 'کمبود سکه', 'coins_needed' => 10];
        }
        
        // کسر سکه
        $userManager->deductCoin($user_id);
        $userManager->deductCoin($user_id); // 10 سکه = 10 بار کسر
        
        // تولید کد منحصر به فرد
        $link_code = 'AL' . strtoupper(substr(md5(uniqid() . $user_id), 0, 10));
        
        // تاریخ انقضا
        $expires_at = date('Y-m-d H:i:s', strtotime("+{$duration_days} days"));
        
        // ذخیره در دیتابیس
        $stmt = $this->db->prepare("
            INSERT INTO anon_links 
            (user_id, link_code, link_name, expires_at) 
            VALUES (?, ?, ?, ?)
        ");
        
        $stmt->execute([$user_id, $link_code, $link_name, $expires_at]);
        $link_id = $this->db->lastInsertId();
        
        // ساخت لینک تلگرام
        $telegram_link = "https://t.me/" . BOT_USERNAME . "?start=anon_{$link_code}";
        
        return [
            'success' => true,
            'link_id' => $link_id,
            'link_code' => $link_code,
            'telegram_link' => $telegram_link,
            'expires_at' => $expires_at,
            'coins_deducted' => 10
        ];
    }
    
    /**
     * دریافت لینک‌های کاربر
     */
    public function getUserLinks($user_id, $active_only = true) {
        $sql = "SELECT * FROM anon_links WHERE user_id = ?";
        
        if ($active_only) {
            $sql .= " AND is_active = TRUE AND (expires_at IS NULL OR expires_at > NOW())";
        }
        
        $sql .= " ORDER BY created_at DESC";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute([$user_id]);
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    /**
     * دریافت پیام‌های لینک
     */
    public function getLinkMessages($link_id, $limit = 50, $offset = 0) {
        $stmt = $this->db->prepare("
            SELECT m.* 
            FROM anon_messages m
            WHERE m.link_id = ? AND m.deleted_at IS NULL
            ORDER BY m.created_at DESC
            LIMIT ? OFFSET ?
        ");
        
        $stmt->execute([$link_id, $limit, $offset]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    /**
     * ذخیره پیام ناشناس
     */
    public function saveAnonymousMessage($link_code, $message, $message_type = 'text', $sender_ip = null) {
        // یافتن لینک
        $stmt = $this->db->prepare("
            SELECT id, user_id, is_active, expires_at 
            FROM anon_links 
            WHERE link_code = ? 
            AND is_active = TRUE
            AND (expires_at IS NULL OR expires_at > NOW())
        ");
        
        $stmt->execute([$link_code]);
        $link = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$link) {
            return ['success' => false, 'error' => 'لینک نامعتبر یا منقضی شده'];
        }
        
        // ذخیره پیام
        $stmt = $this->db->prepare("
            INSERT INTO anon_messages 
            (link_id, sender_ip, message, message_type) 
            VALUES (?, ?, ?, ?)
        ");
        
        $stmt->execute([$link['id'], $sender_ip, $message, $message_type]);
        $message_id = $this->db->lastInsertId();
        
        // آپدیت آمار لینک
        $this->db->prepare("
            UPDATE anon_links 
            SET message_count = message_count + 1, 
                last_message_at = NOW() 
            WHERE id = ?
        ")->execute([$link['id']]);
        
        return [
            'success' => true,
            'message_id' => $message_id,
            'link_owner_id' => $link['user_id']
        ];
    }
    
    /**
     * دریافت آمار لینک
     */
    public function getLinkStats($link_id) {
        $stmt = $this->db->prepare("
            SELECT 
                COUNT(*) as total_messages,
                SUM(CASE WHEN is_read = TRUE THEN 1 ELSE 0 END) as read_messages,
                SUM(CASE WHEN DATE(created_at) = CURDATE() THEN 1 ELSE 0 END) as today_messages,
                MIN(created_at) as first_message,
                MAX(created_at) as last_message
            FROM anon_messages 
            WHERE link_id = ? AND deleted_at IS NULL
        ");
        
        $stmt->execute([$link_id]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
    /**
     * غیرفعال کردن لینک
     */
    public function deactivateLink($link_id, $user_id) {
        $stmt = $this->db->prepare("
            UPDATE anon_links 
            SET is_active = FALSE 
            WHERE id = ? AND user_id = ?
        ");
        
        return $stmt->execute([$link_id, $user_id]);
    }
    
    /**
     * حذف پیام
     */
    public function deleteMessage($message_id, $user_id) {
        // بررسی مالکیت
        $stmt = $this->db->prepare("
            SELECT l.id 
            FROM anon_links l
            JOIN anon_messages m ON m.link_id = l.id
            WHERE m.id = ? AND l.user_id = ?
        ");
        
        $stmt->execute([$message_id, $user_id]);
        
        if ($stmt->fetch()) {
            $stmt = $this->db->prepare("
                UPDATE anon_messages 
                SET deleted_at = NOW() 
                WHERE id = ?
            ");
            
            return $stmt->execute([$message_id]);
        }
        
        return false;
    }
    
    /**
     * بررسی وجود لینک با کد
     */
    public function getLinkByCode($link_code) {
        $stmt = $this->db->prepare("
            SELECT l.*, u.display_name as owner_name
            FROM anon_links l
            LEFT JOIN users u ON u.user_id = l.user_id
            WHERE l.link_code = ? 
            AND l.is_active = TRUE
            AND (l.expires_at IS NULL OR l.expires_at > NOW())
        ");
        
        $stmt->execute([$link_code]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
}
?>